
/*

Code to compute the Kahler potential on the 'symmetric' K3's of hep-th/0506xxx

Designed and written by;
  Matt Headrick (MIT,     headrick@mit.edu)           and, 
  Toby Wiseman  (Harvard, twiseman@fas.harvard.edu)



** Copyright @ June 2005 Matt Headrick and Toby Wiseman **



If you download and use this code in any way, please email us and let
us know so that we can see how accessible the code is, and how useful
putting the code on the web has been.

The code is written in 'C' and can be compiled on linux machines by;

  g++ K3.c

Of course compilers vary and the code might have to be modified on a
particular platform appropriately.

We have commented the code briefly for ease of reading. Obviously
knowledge of C is assumed. The comments are not intended to be full
documentation. The code is simply available so that interested parties
may explicitly see an implementation of the method outlines in
hep-th/0506xxx.

*/



/* Program useage:

The program takes a parameter file 'param' which is a text file of format;

RES KahlerA KahlerB

where RES is an integer from 1-3 giving the resolution (1-low,
2-medium, 3-high) and KahlerA,B are the Kahler moduli in the paper
hep-th/0506xxx a and b resp giving essentially the size of the EH and
torus.

If no param file is found in the working directory, default values for
the above are assumed.

The program will then attempt to compute the Kahler potential for the
Ricci flat metric using Gauss-Seidel iteration of the Monge-Ampere
equation and dynamically determining the value of
sqrt(det(g_{\mu\nu})) to ensure the method converges to a static
solution.

*/

/* The program:

The program stores the Kahler potential in two arrays, K1 and K2 for
the EH and torus patches. This is dones as simply as possible and some
points in the arrays fall outside the coordinate patches defined in
hep-th/0506xxx as;


EH patch:     sigma <= MAXSIG, |y| <= ymax

torus patch:  sigma >= MINSIG


Note! We also require a second EH patch which we term 'K3' (not to be
confused with the manifold!) which covers the w',y' coordinate on
EH. Remember; w = 1/2 z1^2 = w' y'^2 and w' = 1/2 z2^2 = w y^2.

The discrete symmetry; z1 <-> z2 means that K1(w,y) = K3(w,y) and
hence we don't need to actually store an array K3 as it is the same as
K1!

However, suppose we consider a point w,y with sigma<MAXSIG but
|y|>ymax. Then this lies in the w',y' K3 patch rather than the K1
patch, so we will need to consider the patch K3, when interpolating
points at the edges of the K1 and K2 patches, although its potential
will be the same as K1 and so we will not store it or have to relax it
separately.



The discrete symmetries reduce the domains of K1 and K2 to be
simulated. 

Only the upper quadrant in both the complex w, y need be simulated for
K1. 

Similarly for K2 only the upper quadrant in the complex z1 and z2 is
required. However, the symmetries have slightly awkward additional
action on K2, making further identifications;

K2(z1,z2) = K2(z2,z1) = K(\bar{z1},\bar{z2})


and these are built into the code too, complicating it a little.

Whilst K1 is simply stored in an 4-d array for real(w), imag(w),
real(y), imag(y), K2 is stored more cunningly to build in these 3
additional symmetries.

*/



#include "K3.h"      // Header file with variable and function definitions

#include "K3lat.c"   // Nitty gritty routines 


/* Example routines to compute various things once Ricci flat Kalher
   potential is found: Note these routines are not well commented and
   are only included as a simple demonstration for the types of
   geometric quantities that can be computed */

#include "K3process.c" 


/* Main routine that calls everything else */

int main(void)
{
  int iter ;
  long tmp, MAXITER ;
  double vol, maxchange ;

  cout.precision(10) ;

  /* Load in parameters for running the code: in file 'param' */

  getparameters() ;

  /* Allocate memory for the 2 coordinate patches K1 (E-H) and K2 (Torus) */

  initializememory() ;

  /* Initialize template that stores which points are within each
     patch and which are at the edges of the patches */

  inittemp() ; 

  /* Writes discretization info to a file for later analysis purposes */

  initinfo() ;

  /* Computes the analytic value of sqrt(det(g_{\mu\nu})), stored in
     analrootdetg. Initialize the value rootdetg to this value. Note
     rootdetg is updated dynamically to ensure the convergence of the
     Monge-Ampere equation. */

  calcanaldetg() ;

  rootdetg = analrootdetg ;

  /* Initializes the Kahler potentials in the 2 patches to the values
     given in hep-th/0506xxx ; ie. an interpolation from the E-H
     potential to the flat potential */

  initK1() ;
  initK2() ;
  
  /* Make sure points are the edges of patches are correctly set by
     interpolating from the neighbouring patches - not really
     necessary here as initK1,K2 initialize these points but a good
     habit to get into for later!
  */

  fill() ;

  /* Tell user where we are in moduli space and what the analytic
     value for sqrt(det(g)) should be to they can compare with
     dynamically computed value later... */

  cout << "Kahler Moduli (a,b) = ( " << KahlerA << ", " << KahlerB << " ), analytic rootdetg = " << analrootdetg << endl ;

  MAXITER=100000;

  /* Start relaxation process to solve Monge-Ampere equation */

  for(iter=0;iter<MAXITER;iter++) {
      
    /* Tell user how many steps have been taken */
    if(iter%10==0) {      
      cout << endl << iter << " -> " << endl ;
    }
    
    
    /* Update edge points in patches  */

    fill() ;

    /* Do one iteration of Gauss-Seidel on both patches K1, K2. 

       The maximum change in the Kahler potential gives estimate for
       error in discretized equation, so store this in maxchange to
       see if we have reached acceptable error and can stop
       relaxation 
    */

    maxchange = relax(iter%10==0) ;
    
    /* If error small enough then stop relaxation */
  
    if(maxchange<1e-12) {
      cout << "Reached target precision" << endl ;
      iter=MAXITER ;
    }
    
  }
  
  /* Save the Kahler potentials */

  savebin() ;


  /* Compute various bits and pieces from the Ricci-flat metric: These
     routines are all found in K3process and are rather complicated so
     are not commented, but are included simply for illustrative
     purposes.

     Note: the routines involving the Riemann tensor only properly
     work for medium resolution or higher - otherwise there aren't
     enough points to do interpolation they use.

  */

  /* Computes the induced Ricci scalar on the S^2 cycle sigma = 0 */
  calcEHcycle() ;
  /* Computes the induced Ricci scalar on the S^2 cycle z2 = 0 */
  calcToruscycle() ;

  if(RES>=2) {
    /* Computes a 3-dimensional slice of the Kretschmann scalar in the
       z1,z2 coordinates (with Re(z2) = 0) */
    calcRslice3d() ;    
    /* Computes the Euler number directly from the Kahler potential */
    calcEulerTrap() ;
  }
  
  /* Free up the allocated memory upon ending */
  free(K1) ; free(K2) ;
  
}


/* 
   
Routine to perform 1 iteration of Gauss-Seidel to solve the
Monge-Ampere equation in the 2 patches K1 (EH patch) and K2 (Torus
patch).

Returns maximum change in Kahler potential at any point upon updating
-> this gives maximum error remaining in discretized Monge-Ampere
equation.

*/

double relax(bool display)
{
  
  int wre, wim, yre, yim ;
  int z1re, z1im, z2re, z2im ;

  double val, errmax, errmin, change1, change2, old ;

  double avlap1, avlap2 ; 

  double frc, count1, count2 ;

  double dvol ;


  avlap1 = 0. ; count1 = 0. ;
  avlap2 = 0. ; count2 = 0. ;

  /* Perform Gauss-Seidel iteration on all points within K1, EH patch */

  errmax = -1000. ;
  errmin = +1000. ;
  change1 = 0. ;

  /* Loop over all points in K1 array that are potentially inside
     patch. Note that the array goes from 0 -> NW1+4, but two points
     at either end are set by interpolation from neighbouring patches
     and are no in the interior of this patch */

  for(wre=2;wre<NW1+2;wre++) {
    for(wim=2;wim<NW1+2;wim++) {
      for(yre=2;yre<NY1+2;yre++) {
	for(yim=2;yim<NY1+2;yim++) {
	
	  /* Check whether point in array is actually inside the patch */

	  if(abs(coord_y1(yre,yim))>MAXY) continue ;
	  if(2.*abs(coord_w1(wre,wim))*(1.+abs(coord_y1(yre,yim))*abs(coord_y1(yre,yim)))>MAXSIG) continue ;

	  /* Store old value to see how much it is updated by */
	  
	  old = K1[pos1(wre,wim,yre,yim)] ;

	  /* Update point by locally solving the Monge-Ampere
	     equation, computed in UpdateK1_2nd

	     This returns the new value of the Kahler potential and
	     also the value of rootdetg at that point in 'val'
	  */

	  K1[pos1(wre,wim,yre,yim)] = UpdateK1_2nd(wre,wim,yre,yim,&val) ;

	  /* We update the value of rootdetg dynamically by averaging
	     it over the whole manifold. The following code computes
	     this average in the K1 EH patch */

	  /* Firstly only compute the average in the portion of the
	     patch |y1|<1 and sigma<TRANSIG (rather than in full patch
	     |y1|<ymax and sigma < MAXSIG.  Then putting this region
	     together with the appropraite portion of the torus patch
	     and the second EH patch we completely cover the full
	     manifold but do not double cover any regions
	  */

	  /* Only count points in |y1|<1 and sigma<TRANSIG region of patch */

	  if( (abs(coord_y1(yre,yim))<=1.) && (2.*abs(coord_w1(wre,wim))*(1.+abs(coord_y1(yre,yim))*abs(coord_y1(yre,yim)))<=TRANSSIG) ) {
	    dvol = val ; frc = DW1*DW1*DY1*DY1 ;

	    /* Adjust average appropiately if point is at edge of patch */

	    if(wre==2) frc /= 2. ;
	    if(wim==2) frc /= 2. ;
	    if(yre==2) frc /= 2. ;
	    if(yim==2) frc /= 2. ;
	    
	    avlap1 += frc*dvol ; count1 += frc ;
	  }

	  /* avlap1 now contains the contribution from the EH K1 patch
	     to the average of rootdetg. Together with the value from
	     the torus patch these will be used to dynamically
	     determine the new value of rootdetg for the next
	     iteration of Gauss-Seidel. This ensures the procedure
	     converges, and rootdetg can be compared to the analytic
	     value to test how accurate the solution is */

	  /* Compute maximum value of rootdetg g in patch and minimum:
	     Since should be constant this gives indication of how far
	     we are from true solution. */

	  if( val > errmax ) {
	    errmax = val ;
	  }
	  if( val < errmin ) {
	    errmin = val ;
	  }

	  /* Compute maximum update of K1 in G-S iteration */

	  if( fabs(K1[pos1(wre,wim,yre,yim)]-old) > change1 ) {
	    change1 = fabs(K1[pos1(wre,wim,yre,yim)]-old) ;
	  }

	}
      }
    }
  }

  /* Display all the error information for user to see how G-S is progressing */
  
  /* Shows the largest value of rootdetg in K1 patch (errmax) and its
     position at (wreerr1,wimerr1) (yreerr1,yimerr1) 

     shows minimum value of rootdeg, errmin, and its position

     finally shows change1, the maximum update of the potential in the
     K1 patch for the current Gauss-Seidel step */

  if(display) {
    cout << " EH patch;     Max rootdetg: " << errmax << endl ;
    cout << "               Min         : " << errmin << endl ;
    cout << "           Max change in K1: " << change1 << endl ;
  }

  /* Do exactly the same thing for the torus K2 patch */

  errmax = -1000. ;
  errmin = +1000. ;
  change2 = 0. ;

  /* Note that we cycle over all points in K2 array: The last line of
     the loop for z2im builds in the discrete symmetries */

  for(z1re=2; z1re<NZ+2; z1re++) {
    for(z1im=2; z1im<NZ+2; z1im++) { 
      for(z2re=2; z2re<NZ+2; z2re++) {
	for(z2im=max(z1re,z1im); z2im<NZ+2; z2im++) {
	  
	  /* Carry one if point in K2 array is actually in the
	     patch... */

	  if( tempK2[K2pos[pos2(z1re,z1im,z2re)]+z2im] != 0 ) continue ;

	  /* Same as for EH patch above; Update K2 locally using
	     G-S. Store rootdetg locally there in val. */	  

	  old = K2[K2pos[pos2(z1re,z1im,z2re)]+z2im] ;

	  K2[K2pos[pos2(z1re,z1im,z2re)]+z2im] = UpdateK2_2nd(z1re,z1im,z2re,z2im,&val) ;
	  
	  /* Keep running average of rootdetg. */

	  if( (abs(coord_z1(z1re,z1im))*abs(coord_z1(z1re,z1im)) + abs(coord_z2(z2re,z2im))*abs(coord_z2(z2re,z2im)))>=TRANSSIG ) {
	    dvol = val ; frc = DZ*DZ*DZ*DZ ;
	    
	    if(z2im==z1re) frc /= 2. ;
	    if(z2im==z1im) frc /= 2. ;
	    
	    if(z1re==2) frc /= 2. ;
	    if(z1im==2) frc /= 2. ;
	    if(z2re==2) frc /= 2. ;
	    if(z2im==2) frc /= 2. ;
	    
	    if(z1re==NZ-1+2) frc /= 2. ;
	    if(z1im==NZ-1+2) frc /= 2. ;
	    if(z2re==NZ-1+2) frc /= 2. ;
	    if(z2im==NZ-1+2) frc /= 2. ;

	    avlap2 += frc*dvol ; count2 += frc ;
	  }

	  /* Track max and min rootdetg and max update of K2 as for EH
	     patch above */

	  if( val > errmax ) {
	    errmax = val ;
	  }
	  if( val < errmin ) {
	    errmin = val ;
	  }
	  if( fabs(K2[K2pos[pos2(z1re,z1im,z2re)]+z2im]-old) > change2 ) {
	    change2 = fabs(K2[K2pos[pos2(z1re,z1im,z2re)]+z2im]-old) ;
	  }

	}
      }
    }
  }

  if(display) {
    cout << " Torus patch;  Max rootdetg: " << errmax << endl ;
    cout << "               Min         : " << errmin << endl ;
    cout << "           Max change in K2: " << change2 << endl ;
  }

  /* Compute volume average of rootdetg to dynamically update it for
     next iteration */


  /* Volume average rootdetg 

  coord vol patch 1 = 16.*2.*(16.*count1)
  
      represent 1/16 of the full w, y domain.

      2 patches.

      16. EH's in total

  coord vol patch 2 = (256./2.)*(8.*count2)
  
      represent 1/8 of the full z1, z2 domain.

      4^4 domains in full K3 but 1/2 from the Z2 orbifold

  so total ~ const*( count1 + 2.*count2 )      (with const = 2.*256.)

  */

  rootdetg = ( avlap1 + 2.*avlap2 )/( count1 + 2.*count2 ) ;

  /* Show user the dynamically updated rootdetg */

  if(display) {

    cout << "         Averaged rootdetg = " << rootdetg << "   (analytic value = " << analrootdetg << ")" << endl ;
    
  }

  /* Return the maximum update of potential over the two patches */

  return(max(change1,change2)) ;

}



/* This routine computes the analytic value of rootdetg from the
   position in moduli space. This is then to be compared with the
   dynamically determined one found whilst solving the Monge Ampere
   equation. As the continuum is approached these should coincide. If
   one uses the analytic value in the Monge-Ampere equation the
   solution will never reach a static fixed point, but will slowly
   drift by a constant Kahler transform. This is fine, as the drifting
   gets less with increasing resolution, but we dynamically determine
   rootdetg so that the M-A equation is solved numerically exactly
   given the particular resolution, and there is no drifiting. Either
   method is as good as the other really. */

void calcanaldetg()
{
  double a, t ;
  
  a = KahlerA ;
  t = KahlerB ;

  analrootdetg = 0.25*pow(t,4) - 2.*pow(PI,2)*pow(a,4) ;

}



/* This code sets up the potential in the EH patch to interpolate
   between the EH potential (with the correct period of J over the
   S^2, given by the kahler modulus a) for sigma<=TRANSSIG to the flat
   torus potential for sigma>TRANSSIG */

void initK1(void) 
{
  int w1re, w1im, y1re, y1im ;
  complex<double> w1, y1 ;

  double absw1, absy1, sig, sig0, Kval, KEH ;

  double k, a, t, b, c ;

  a    = KahlerA ;
  t    = KahlerB ;
  sig0 = TRANSSIG ;

  k = (6*pow(a,2) - 8*log(sig0)*pow(a,2) + 3*sig0*pow(t,2))/8. ;

  b = (pow(sig0,-2)*(-4*pow(a,2) + 3*sig0*pow(t,2)))/4. ;

  c = (pow(sig0,-4)*(2*pow(a,2) - sig0*pow(t,2)))/8. ;


  for(w1re=0; w1re<NW1+4; w1re++) {
    for(w1im=0; w1im<NW1+4; w1im++) {
      
      for(y1re=0; y1re<NY1+4; y1re++) {
	for(y1im=0; y1im<NY1+4; y1im++) {
	
	  w1 = coord_w1(w1re,w1im) ;
	  y1 = coord_y1(y1re,y1im) ;

	  absw1 = abs(w1) ;
	  absy1 = abs(y1) ;


	  sig  = 2.*absw1*(1 + absy1*absy1) ;
	  
	  if(sig<=TRANSSIG) {

	    Kval = 0.5*( k + pow(a,2)*log(1.+absy1*absy1) + b*pow(sig,2) + c*pow(sig,4) ) ;

	  } else {

	    Kval = 0.5*pow(t,2)*sig ;

	  }

	  K1[pos1(w1re,w1im,y1re,y1im)] = Kval ;

	}
      }
      
    }
  }
  
}

/* This code sets up the potential in the torus patch to be the flat
   one with the correct kahler modulus b */

void initK2(void) 
{
  int z1re, z1im, z2re, z2im ;
  complex<double> z1, z2 ;

  double absz1, absz2, sig, Kval, KEH ;

  double t ;

  t = KahlerB ;

  for(z1re=0; z1re<NZ+4; z1re++) {
    for(z1im=0; z1im<NZ+4; z1im++) { 
      for(z2re=0; z2re<NZ+4; z2re++) { 
	for(z2im=max(z1re,z1im); z2im<NZ+4; z2im++) { 

	  z1 = coord_z1(z1re,z1im) ;
	  z2 = coord_z2(z2re,z2im) ;

	  absz1 = abs(z1) ;
	  absz2 = abs(z2) ;

	  sig  = absz1*absz1 + absz2*absz2 ;

	  Kval = 0.5*pow(t,2)*sig ;

	  K2[K2pos[pos2(z1re,z1im,z2re)]+z2im] = Kval ;

	}
      }
    }
  }
  
}


/* Routine used by Gauss-Seidel relaxation to compute the discretized
   Monge-Ampere equation at a point in the EH K2 patch and return the
   new value of the Kahler potential to locally satisfy the
   equation. Second order differencing is used.

   As a by-product the routine also returns (in `lap') the local value
   of rootdetg so that this can be dynamically averaged.

*/


double UpdateK1_2nd(int w1re, int w1im, int y1re, int y1im, double *lap)
{

  double newK1, Kww, Kyy, Kww2, Kyy2, Kwy1, Kwy2, Kwy3, Kwy4, src ;

  double K0000, Ku000, Kd000, K0u00, K0d00, K00u0, K00d0, K000u, K000d ;
  double Ku0u0, Kd0u0, Ku0d0, Kd0d0 ;
  double Ku00u, Kd00u, Ku00d, Kd00d ;
  double K0uu0, K0du0, K0ud0, K0dd0 ;
  double K0u0u, K0d0u, K0u0d, K0d0d ;


  K0000 = K1[pos1(w1re,w1im,y1re,y1im)] ;

  Ku000 = K1[pos1(w1re+1,w1im,y1re,y1im)] ;
  Kd000 = K1[pos1(w1re-1,w1im,y1re,y1im)] ;

  K0u00 = K1[pos1(w1re,w1im+1,y1re,y1im)] ;
  K0d00 = K1[pos1(w1re,w1im-1,y1re,y1im)] ;

  K00u0 = K1[pos1(w1re,w1im,y1re+1,y1im)] ;
  K00d0 = K1[pos1(w1re,w1im,y1re-1,y1im)] ;

  K000u = K1[pos1(w1re,w1im,y1re,y1im+1)] ;
  K000d = K1[pos1(w1re,w1im,y1re,y1im-1)] ;

  Ku0u0 = K1[pos1(w1re+1,w1im,y1re+1,y1im)] ;
  Kd0u0 = K1[pos1(w1re-1,w1im,y1re+1,y1im)] ;
  Ku0d0 = K1[pos1(w1re+1,w1im,y1re-1,y1im)] ;
  Kd0d0 = K1[pos1(w1re-1,w1im,y1re-1,y1im)] ;
  
  Ku00u = K1[pos1(w1re+1,w1im,y1re,y1im+1)] ;
  Kd00u = K1[pos1(w1re-1,w1im,y1re,y1im+1)] ;
  Ku00d = K1[pos1(w1re+1,w1im,y1re,y1im-1)] ;
  Kd00d = K1[pos1(w1re-1,w1im,y1re,y1im-1)] ;

  K0uu0 = K1[pos1(w1re,w1im+1,y1re+1,y1im)] ;
  K0du0 = K1[pos1(w1re,w1im-1,y1re+1,y1im)] ;
  K0ud0 = K1[pos1(w1re,w1im+1,y1re-1,y1im)] ;
  K0dd0 = K1[pos1(w1re,w1im-1,y1re-1,y1im)] ;

  K0u0u = K1[pos1(w1re,w1im+1,y1re,y1im+1)] ;
  K0d0u = K1[pos1(w1re,w1im-1,y1re,y1im+1)] ;
  K0u0d = K1[pos1(w1re,w1im+1,y1re,y1im-1)] ;
  K0d0d = K1[pos1(w1re,w1im-1,y1re,y1im-1)] ;


  Kww  = ( Ku000 + Kd000 + K0u00 + K0d00 - 4.*K0000 ) ;
  Kyy  = ( K00u0 + K00d0 + K000u + K000d - 4.*K0000 ) ;
  

  Kww2  = ( Ku000 + Kd000 + K0u00 + K0d00 ) ;

  Kyy2  = ( K00u0 + K00d0 + K000u + K000d ) ;

  Kwy1 = ( Ku0u0 - Ku0d0 - Kd0u0 + Kd0d0 )/4. ;

  Kwy2 = ( Ku00u - Ku00d - Kd00u + Kd00d )/4. ;

  Kwy3 = ( K0uu0 - K0ud0 - K0du0 + K0dd0 )/4. ;
  
  Kwy4 = ( K0u0u - K0u0d - K0d0u + K0d0d )/4. ;
  

  src  = (Kwy1 + Kwy4)*(Kwy1 + Kwy4) + (Kwy2 - Kwy3)*(Kwy2 - Kwy3) ;

  *lap = ( Kww*Kyy - src )/16./(DW1*DW1*DY1*DY1) ;

  newK1 = ( Kww2 + Kyy2 - sqrt( 4.*(src + 16*DW1*DW1*DY1*DY1*rootdetg ) + (Kww2-Kyy2)*(Kww2-Kyy2) ) )/8. ;

  return( newK1 ) ;

}


/* Same as above but for torus K2 patch. This is a little more
   complicated as the discrete symmetries of K2 must be
   implemented. */


double UpdateK2_2nd(int z1re, int z1im, int z2re, int z2im, double *lap)
{

  double newK2, Kww2, Kyy2, Kww, Kyy, Kwy1, Kwy2, Kwy3, Kwy4, src ;

  double K0000 ;
  double Ku000, Kd000, K0u00, K0d00, K00u0, K00d0, K000u, K000d ;
  double Ku0u0, Kd0u0, Ku0d0, Kd0d0 ;
  double Ku00u, Kd00u, Ku00d, Kd00d ;
  double K0uu0, K0du0, K0ud0, K0dd0 ;
  double K0u0u, K0d0u, K0u0d, K0d0d ;

  int i1, i2, i3, i4, tmpz1re, tmpz1im, tmpz2re, tmpz2im ;

  /* Put values of K2 used in local Monge-Ampere equation in B array */

  for(i1=-1;i1<=1;i1++) {
    for(i2=-1;i2<=1;i2++) {
      for(i3=-1;i3<=1;i3++) {
	for(i4=-1;i4<=1;i4++) {

	  /* Only store values that are actually used in second order
	     differenced Monge-Ampere equation */

	  if(abs(i1)+abs(i2)>1) continue ;
	  if(abs(i3)+abs(i4)>1) continue ;
	  
	  tmpz1re = z1re+i1 ;
	  tmpz1im = z1im+i2 ;
	  tmpz2re = z2re+i3 ;
	  tmpz2im = z2im+i4 ;

	  /* If point lies outside the fundamental domain under the
	     additional discrete symmetries that K2 satisfies, then
	     map back to fundamental domain using the function
	     reflectZ */

	  reflectZ(&tmpz1re,&tmpz1im,&tmpz2re,&tmpz2im) ;

	  /* Now point (tmpz1re,tmpz1im,tmpz2re,tmpz2im) does lie in
	     fundamental domain get value of K2 there */

	  B[i1+2][i2+2][i3+2][i4+2] = K2[K2pos[pos2(tmpz1re,tmpz1im,tmpz2re)]+(tmpz2im)] ;


	}
      }
    }
  }

  /* Now have all points required by equation go and compute
     Monge-Ampere equation and the value of K2 that will satisfy it
     locally */

  K0000 = B[2][2][2][2] ;

  Ku000 = B[3][2][2][2] ;
  Kd000 = B[1][2][2][2] ;

  K0u00 = B[2][3][2][2] ;
  K0d00 = B[2][1][2][2] ;

  K00u0 = B[2][2][3][2] ;
  K00d0 = B[2][2][1][2] ;

  K000u = B[2][2][2][3] ;
  K000d = B[2][2][2][1] ;

  Ku0u0 = B[3][2][3][2] ;
  Kd0u0 = B[1][2][3][2] ;
  Ku0d0 = B[3][2][1][2] ;
  Kd0d0 = B[1][2][1][2] ;
  
  Ku00u = B[3][2][2][3] ;
  Kd00u = B[1][2][2][3] ;
  Ku00d = B[3][2][2][1] ;
  Kd00d = B[1][2][2][1] ;

  K0uu0 = B[2][3][3][2] ;
  K0du0 = B[2][1][3][2] ;
  K0ud0 = B[2][3][1][2] ;
  K0dd0 = B[2][1][1][2] ;

  K0u0u = B[2][3][2][3] ;
  K0d0u = B[2][1][2][3] ;
  K0u0d = B[2][3][2][1] ;
  K0d0d = B[2][1][2][1] ;


  Kww  = ( Ku000 + Kd000 + K0u00 + K0d00 - 4.*K0000 ) ;
  Kyy  = ( K00u0 + K00d0 + K000u + K000d - 4.*K0000 ) ;
 

  Kww2  = ( Ku000 + Kd000 + K0u00 + K0d00 ) ;
  
  Kyy2  = ( K00u0 + K00d0 + K000u + K000d ) ;


  Kwy1 = ( Ku0u0 - Ku0d0 - Kd0u0 + Kd0d0 )/4. ;

  Kwy2 = ( Ku00u - Ku00d - Kd00u + Kd00d )/4. ;
  
  Kwy3 = ( K0uu0 - K0ud0 - K0du0 + K0dd0 )/4. ;
  
  Kwy4 = ( K0u0u - K0u0d - K0d0u + K0d0d )/4. ;
  

  src  = (Kwy1 + Kwy4)*(Kwy1 + Kwy4) + (Kwy2 - Kwy3)*(Kwy2 - Kwy3) ;

  *lap = ( Kww*Kyy - src )/16./(DZ*DZ*DZ*DZ) ;

  newK2 = ( Kww2 + Kyy2 - sqrt( 4.*(src + 16*DZ*DZ*DZ*DZ*rootdetg ) + (Kww2-Kyy2)*(Kww2-Kyy2) ) )/8. ;

  return( newK2 ) ;

}


/* For every point in the K1 and K2 array, store a value in tempK1 and
   tempK2 which is;

   0 - if the point is in the interior of the coordinate patch

   1 - if the point is outside the patch, but will be required to
   evaluate the Monge-Ampere equation for a point in the interior near
   the edge.

   -1 - if point is outside the coordinate patch altogether and is
    never required.


   The routine 'fill' below when called updates all points with value
   '1' by interpolating from the neighbouring patches. This is called
   before the Monge-Ampere equation is solved to ensure that points at
   the edges of patches are computed correctly.

*/


void inittemp(void)
{
  int wre,wim,yre,yim ;
  int z1re,z1im,z2re,z2im ;

  double val, sig, sigy ;

  int i1,i2,i3,i4 ;

  int tmpz1re,tmpz1im,tmpz2re,tmpz2im ;

  /* Initialize all points with -1 (never required) */

  for(wre=0;wre<NW1+4;wre++) {
    for(wim=0;wim<NW1+4;wim++) {
      for(yre=0;yre<NY1+4;yre++) {
	for(yim=0;yim<NY1+4;yim++) {
	  
	  tempK1[pos1(wre,wim,yre,yim)] = -1  ;
	  
	}
      }
    }
  }

  /* For every point in the interior of the patch, compute which
     points are required to evaluate the Monge Ampere equation there
     are store value '1' in template array */

  for(wre=2;wre<NW1+2;wre++) {
    for(wim=2;wim<NW1+2;wim++) {
      for(yre=2;yre<NY1+2;yre++) {
	for(yim=2;yim<NY1+2;yim++) {

	  sigy = abs(coord_y1(yre,yim)) ;

	  sig = ( 2.*abs(coord_w1(wre,wim))*(1.+abs(coord_y1(yre,yim))*abs(coord_y1(yre,yim)))) ;

	  /* If point is in interior... */

	  if( (sig<=MAXSIG) && (sigy<=MAXY) ) {

	    /* ... go through all points used by second order
	       differenced Monge-Ampere equation ... */

	    for(i1=-1;i1<=1;i1++) {
	      for(i2=-1;i2<=1;i2++) {
		for(i3=-1;i3<=1;i3++) {
		  for(i4=-1;i4<=1;i4++) {
		    
		    if(abs(i1)+abs(i2)>1) continue ;
		    if(abs(i3)+abs(i4)>1) continue ;

		    /* ... and store '1' in template array */

		    tempK1[pos1(wre+i1,wim+i2,yre+i3,yim+i4)] = 1  ;
		    
		  }
		}
	      }
	    }

	  } 

	}
      }
    }
  }

  /* Now go through and store '0' in all points that are actually in
     the interior of the patch */

  for(wre=2;wre<NW1+2;wre++) {
    for(wim=2;wim<NW1+2;wim++) {
      for(yre=2;yre<NY1+2;yre++) {
	for(yim=2;yim<NY1+2;yim++) {
	  
	  sigy = abs(coord_y1(yre,yim)) ;

	  sig = ( 2.*abs(coord_w1(wre,wim))*(1.+abs(coord_y1(yre,yim))*abs(coord_y1(yre,yim)))) ;

	  if( (sig<=MAXSIG) && (sigy<=MAXY) ) {

	    tempK1[pos1(wre,wim,yre,yim)] = 0  ;
	    
	  } 

	}
      }
    }
  }


  /* Do exactly the same thing for the torus K2 patch, but taking into
     account the additional discrete symmetries */

  for(z1re=0; z1re<NZ+4; z1re++) {
    for(z1im=0; z1im<NZ+4; z1im++) { 
      for(z2re=0; z2re<NZ+4; z2re++) {
        for(z2im=max(z1re,z1im); z2im<NZ+4; z2im++) {
	  
	  tempK2[K2pos[pos2(z1re,z1im,z2re)]+z2im] = -1  ;

	}
      }
    }
  }

  for(z1re=2; z1re<NZ+2; z1re++) {
    for(z1im=2; z1im<NZ+2; z1im++) { 
      for(z2re=2; z2re<NZ+2; z2re++) {
        for(z2im=max(z1re,z1im); z2im<NZ+2; z2im++) {
  
	  sig = abs(coord_z1(z1re,z1im))*abs(coord_z1(z1re,z1im)) + abs(coord_z2(z2re,z2im))*abs(coord_z2(z2re,z2im)) ;

	  if( (sig>=MINSIG) ) {

	    /* For every point in interior, go through points required
	       by Monge-Ampere equation */

	    for(i1=-1;i1<=1;i1++) {
	      for(i2=-1;i2<=1;i2++) {
		for(i3=-1;i3<=1;i3++) {
		  for(i4=-1;i4<=1;i4++) {
		    
		    if(abs(i1)+abs(i2)>1) continue ;
		    if(abs(i3)+abs(i4)>1) continue ;

		    tmpz1re = z1re+i1 ;
		    tmpz1im = z1im+i2 ;
		    tmpz2re = z2re+i3 ;
		    tmpz2im = z2im+i4 ;

		    /* Remember that this point may fall outside
		       fundamental domain of discrete symmetries, so
		       if it does, map it back */

		    reflectZ(&tmpz1re,&tmpz1im,&tmpz2re,&tmpz2im) ;
		    
		    tempK2[K2pos[pos2(tmpz1re,tmpz1im,tmpz2re)]+tmpz2im] = 1  ;

		  }
		}
	      }
	    }


	  }

	}
      }
    }
  }

  for(z1re=2; z1re<NZ+2; z1re++) {
    for(z1im=2; z1im<NZ+2; z1im++) { 
      for(z2re=2; z2re<NZ+2; z2re++) {
        for(z2im=max(z1re,z1im); z2im<NZ+2; z2im++) {
	  
	  sig = abs(coord_z1(z1re,z1im))*abs(coord_z1(z1re,z1im)) + abs(coord_z2(z2re,z2im))*abs(coord_z2(z2re,z2im)) ;

	  if( (sig>=MINSIG) ) {
	    tempK2[K2pos[pos2(z1re,z1im,z2re)]+z2im] = 0  ;
	  }

	}
      }
    }
  }
  
  cout << "Done" << endl ;

}



/* Go through all points and if they have value '1' in their template
   array call the getK1, getK2 functions which will call the necessary
   interpolation routines to get the potential for the neighbouring
   patches */

void fill(void)
{

  int wre,wim,yre,yim ;
  int z1re,z1im,z2re,z2im ;

  for(wre=0;wre<NW1+4;wre++) {
    for(wim=0;wim<NW1+4;wim++) {
      for(yre=0;yre<NY1+4;yre++) {
	for(yim=0;yim<NY1+4;yim++) {
	
	  if( tempK1[pos1(wre,wim,yre,yim)] == 1 ) {
	    K1[pos1(wre,wim,yre,yim)] = getK1(wre,wim,yre,yim) ;
	  }

	}
      }
    }
  }

  for(z1re=0; z1re<NZ+4; z1re++) {
    for(z1im=0; z1im<NZ+4; z1im++) {
      for(z2re=0; z2re<NZ+4; z2re++) {
        for(z2im=max(z1re,z1im); z2im<NZ+4; z2im++) { 

	if( tempK2[K2pos[pos2(z1re,z1im,z2re)]+z2im] == 1 )
	    K2[K2pos[pos2(z1re,z1im,z2re)]+z2im] = getK2(z1re,z1im,z2re,z2im) ;
	  
	}
      }
    }
  }
  
}


/* Routine to read in file 'param' and then setup the geometry of the
   patches and discretization appropriately. If no file is specified
   then use default values. */

void getparameters(void)
{
  ifstream param("param") ;

  if(param==NULL) {

    cout << "No input parameter file found" << endl ;
    
    RES = 1 ;
    
    KahlerA = 0.1 ;

    KahlerB = 1.0 ;


  } else {

    cout << "Reading input parameter file" << endl ;

    param >> RES ;

    param >> KahlerA ;

    param >> KahlerB ;

    param.close() ;

  }

  switch(RES) {
  case(1) : cout << "Geom1: Low res" << endl ; break ;
  case(2) : cout << "Geom1: Med res" << endl ; break ;
  case(3) : cout << "Geom1: High res" << endl ; break ;
  default :
    cout << "Error reading RES = " << RES << endl ;
    exit(1) ;
  }
  
  if(KahlerA<0. || KahlerA>10.0) {
    cout << "Error reading KahlerA = " << KahlerA << endl ;
    exit(1) ;
  } else {
    cout << "KahlerA = " << KahlerA << endl ;
  }

  if(KahlerB<0. || KahlerB>10.0) {
    cout << "Error reading KahlerB = " << KahlerB << endl ;
    exit(1) ;
  } else {
    cout << "KahlerB = " << KahlerB << endl ;
  }

  if(pow(KahlerB,2) - 4.*PI*pow(KahlerA,2)<0) {
    cout << "*** Warning ***  Outside the Kahler cone you will get rubbish!" << endl ;
  }


  /* Now set up the variables! */

  switch(RES) {

    /*  **** Geometry 1: ****
	
    w: 0 -> 0.3  ie. sigmax = 0.6
    
    y: 0 -> 1.25

    z: 0 -> 1

    MAXY      = 1.25 ;
    MINSIG    = 0.32 ;
    TRANSSIG  = 0.5 ;
    MAXSIG    = 0.600000001 ;

    */

  case(1) :

    // Low res:

    INTERP  = 1 ;

    NW1     = 7 ;
    DW1     = 0.05 ; 

    NY1     = 6 ;
    DY1     = 0.25 ;

    NZ      = 11 ;
    DZ      = 0.10 ; 

    MAXY      = 1.2500000001 ;
    MINSIG    = 0.3200000001 ;
    TRANSSIG  = 0.5000000001 ;
    MAXSIG    = 0.6000000001 ;

    break ;
    
  case(2) :
    
    // Med res

    INTERP  = 3 ;

    NW1     = 13 ;
    DW1     = 0.025 ;

    NY1     = 11 ;
    DY1     = 0.125 ;

    NZ      = 21 ;
    DZ      = 0.05 ;

    MAXY      = 1.2500000001 ;
    MINSIG    = 0.3200000001 ;
    TRANSSIG  = 0.5000000001 ;
    MAXSIG    = 0.6000000001 ;

    break ;
    
  case(3) :

    // High res

    INTERP  = 3 ;
    
    NW1     = 25 ;
    DW1     = 0.0125 ;

    NY1     = 21 ;
    DY1     = 0.0625 ;

    NZ      = 41 ;
    DZ      = 0.025 ;

    MAXY      = 1.2500000001 ;
    MINSIG    = 0.3200000001 ;
    TRANSSIG  = 0.5000000001 ;
    MAXSIG    = 0.6000000001 ;
    
   break ;
   
  default :

    cout << "RES incorrect (" << RES << ")" << endl ;

    exit(1) ;

  }

  /* Rescale so z 0->1 */

  DW1 *= pow(0.25,2) ;
  DZ  *= 0.25 ;

  MINSIG   *= pow(0.25,2) ;
  TRANSSIG *= pow(0.25,2) ;
  MAXSIG   *= pow(0.25,2) ;

  /* So... */

  K1dimA = (4 + NW1) ;
  K1dimB = (4 + NW1) ;
  K1dimC = (4 + NY1) ;
  K1dimD = (4 + NY1) ;
  
  K2dimA = (4 + NZ) ;
  K2dimB = (4 + NZ) ;
  K2dimC = (4 + NZ) ;
  K2dimD = (4 + NZ) ;
  
  K1A = (K1dimB*K1dimC*K1dimD) ;
  K1B =        (K1dimC*K1dimD) ;
  K1C =               (K1dimD) ;
  
  K2A = (K2dimB*K2dimC) ;
  K2B =        (K2dimC) ;
 
}




/* Routine to save discretization information to file 'info' for later
   convenience when analysing data */


void initinfo(void)
{

  ofstream info("info") ;

  info << NW1 << " " << NY1 << " " << DW1 << " " << DY1 << endl ;

  info << NZ  << " " << DZ  << endl ;

  info.close() ;

}


/* Routine to save the values of K1 and K2 in the two patches to files
   'saveK1' and 'saveK2' so that they can be used later for
   analysis. Note data is stored in binary format to avoid creating
   huge files.
*/

void savebin(void)
{
  int wre,wim,yre,yim ;
  int z1re,z1im,z2re,z2im ;
  double val ;

  FILE *savefileK1, *savefileK2 ;

  savefileK1 = fopen("saveK1","w") ;
  savefileK2 = fopen("saveK2","w") ;

  if(savefileK1==NULL || savefileK2==NULL) {

    cout << "File error saving!" << endl ;
    exit(1) ;

  }

  cout << "Saving K1:  " ;

  for(wre=2;wre<NW1+2;wre++) {
    cout << "." ; cout.flush() ;
    for(wim=2;wim<NW1+2;wim++) {
      for(yre=2;yre<NY1+2;yre++) {
	for(yim=2;yim<NY1+2;yim++) {

	  if( abs(coord_y1(yre,yim))>MAXY || ( 2.*abs(coord_w1(wre,wim))*(1.+abs(coord_y1(yre,yim))*abs(coord_y1(yre,yim))) > MAXSIG ) )  {

	    val = 0. ;

	  } else {

	    val = getK1(wre,wim,yre,yim) ;

	  }

	  fwrite(&val,sizeof(val),1,savefileK1) ;

	}
      }
    }
  }
  
  cout << endl << "       K2:  " ;

  for(z1re=2;z1re<NZ+2;z1re++) {
    cout << "." ; cout.flush() ;
    for(z1im=2;z1im<NZ+2;z1im++) {
      for(z2re=2;z2re<NZ+2;z2re++) {
	for(z2im=max(z1re,z1im);z2im<NZ+2;z2im++) {

	  if( ( pow(abs(coord_z1(z1re,z1im)),2) + pow(abs(coord_z2(z2re,z2im)),2) ) < MINSIG ) {
	    val = 0 ;
	  } else {
	    val = getK2(z1re,z1im,z2re,z2im) ;
	  }

	  fwrite(&val,sizeof(val),1,savefileK2) ;
	}
      }
    }
  }

  fclose(savefileK1) ;
  fclose(savefileK2) ;
  
  cout << endl << "Done" << endl ;

}

