/*
 * File:   TseGraph.cpp
 * Author: T.S.Evans, Physics Dept., Imperial College London
 * TseGraph is for weighted or unweighted, undirected  graphs.
 * Much of the code is OK for directed graphs but this is not fully implemented.
 * See TseGraph.h for more information.
 * Please acknowledge use of this code if you use this code.
 * 22nd December, 2009
 */

#include "TseGraph.h"

using namespace std;

TseGraph::TseGraph(){
weightedGraph=false;
directedGraph=false;
}

/**
 *Sets initial size and graph to be undirected and unweighted.
 */
TseGraph::TseGraph(int numberVertices, int numberStubs){
 //setSize(numberVertices, numberStubs);
 weightedGraph=false;
 directedGraph=false;
 }

///**
// * Allocate initial memory but this can be exanded as we go.
// * This gives an automatic number of vertices, which are set to have negative stub numbers.
// * However this then screws up number of vertex counts.
// */
//void
//TseGraph::setSize(int numberVertices, int numberStubs){
//// would like to allocate memory but not sure how.
//// vertexToStub(numberVertices);
////  stubToVertex(numberStubs);
//    if (vertexToStub.size()<numberVertices)
//                        vertexToStub.resize(numberVertices, -97531);
// }

///**
// * Allocate initial memory but this can be exanded as we go.
// * This gives an automatic number of vertices, which are set to have negative stub numbers.
// * However this then screws up number of vertex counts.
// */
//void
//TseGraph::setSize(int numberVertices){
//    if (vertexToStub.size()<numberVertices)
//                        vertexToStub.resize(numberVertices);
// }


void
TseGraph::read(char *filename, bool weightsOn, bool directedOn){

  cout << "Starting to read file " << filename << endl;
  ifstream finput;
  finput.open(filename,fstream::in);
  if (!finput) {
      cerr << "Can't open input file " << filename << endl;
      exit(1);
  }


  weightedGraph = weightsOn;
  directedGraph = directedOn;
  unsigned int lineNumber=0;

  int source , target, weight=-1;

  while (!finput.eof()) {
    lineNumber++;  
    source , target, weight=-1; // problem with last line of file
    if (weightedGraph)
      finput >> source >> target >> weight;
    else
      finput >> source >> target;

    if ((source<0) || (target<0)) cerr << "!!! Warning line number " << lineNumber << " ignored: source " << source <<", target " << target << endl;
    else if (finput) {// finput test deals with end of file issues
            //cout << source << " - " << target << "\n";
         addEdgeSlow(source, target, weight);}
    
    if (getNumberStubs()%1000000==0) {cerr << "."; fflush(stderr);}
    if (getNumberStubs()%10000000==0) {cerr << getNumberVertices() << "\n"; fflush(stderr);}

  }

  finput.close();
  cout << "\nFinished reading file " << filename << endl;


}

void
TseGraph::write() {
    write(cout, true);
}


void
TseGraph::write(char *outFile) {
  ofstream fout(outFile);
  if (!fout) {
      cerr << "Can't open output file " << outFile << endl;
      exit(1);
  }
  write(fout) ;
  fout.close();
}

void
TseGraph::write(ostream & fout) {write(fout, false);}

// needs ostream for output (e.g. cout) and true (false) if want header line
void
TseGraph::write(ostream & fout, bool labelOn) {
 int source,target=-1;
 if (labelOn) {
    if (directedGraph) fout << "s \tt";
    else fout << "v1 \t v2";
    if (weightedGraph) fout << "\tw";
    fout <<endl;
 }
 for (int stub=0; stub<getNumberStubs(); stub++)
 {
      source=stubToVertex[stub++];
      target=stubToVertex[stub];
      fout << source << "\t" << target;
      if (weightedGraph) fout << "\t" <<  edgeWeight[stub>>1];
      fout << endl;
 }
}


bool
TseGraph::isWeighted(){return weightedGraph;}

bool
TseGraph::isDirected(){return directedGraph;}

void
TseGraph::setWeighted(bool b){weightedGraph=b;}

void
TseGraph::setDirected(bool b){directedGraph=b;}
//if (b) {cerr <<"Currently only undirected graphs allowed for TseGraph.\n"; exit(0);}

bool 
TseGraph::check(){
 return  (edgeWeight.size()*2 == stubToVertex.size());
 } 
  

int
TseGraph::getNumberStubs(){return stubToVertex.size();}

int
TseGraph::getNumberVertices(){return vertexToStub.size();}

int
TseGraph::getVertexDegree(int v){return vertexToStub[v].size();}

int
TseGraph::getVertexOutDegree(int v){return vertexToStub[v].size();}

int
TseGraph::getVertexInDegree(int v){return vertexToIncomingStub[v].size();}

/**
 * Finds the strength of a vertex.
 * Works for weighted and unweighted graphs.
 * If directed this is the out strength.
 */
double
TseGraph::getVertexStrength(int v){
    int k=getVertexDegree(v);
    if (!weightedGraph) return k;
    double s=0;
    for (int n=0; n<k; n++) s+=edgeWeight[(vertexToStub[v][n]>>1)];
    return s;
    }

/**
 * Finds the out-strength of a vertex.
 * Works for weighted and unweighted graphs,
 * directed and undirected.
 */
double
TseGraph::getVertexOutStrength(int v){return getVertexStrength(v);}


/**
 * Finds the in-strength of a vertex.
 * Works only for directed graphs but
 * works for both weighted and unweighted graphs.
 */
double
TseGraph::getVertexInStrength(int v){
    int k=getVertexInDegree(v);
    if (!weightedGraph) return k;
    double s=0;
    for (int n=0; n<k; n++) s+=edgeWeight[(vertexToIncomingStub[v][n]>>1)];
    return s;
    }

/**
 * Gets index of n-th stub of vertex v.
 * No Tests done.
 * 0<=n<degree(v).
 */
int
TseGraph::getStub(int v, int n){
	return vertexToStub[v][n];
}

/**
 * Gets index of n-th outgoing stub of vertex v.
 * No Tests done. Works for directed and undirected graphs.
 * 0<=n<degree(v).
 */
int
TseGraph::getOutStub(int v, int n){
	return vertexToStub[v][n];
}

/**
 * Gets index of n-th outgoing stub of vertex v.
 * No Tests done. Works for directed graphs only.
 * 0<=n<degree(v).
 */
int
TseGraph::getInStub(int v, int n){
	return vertexToIncomingStub[v][n];
}

/**
 * Gets index of other end of n-th stub of vertex v.
 * No Tests done. Works for directed and undirected graphs.
 * 0<=n<degree(v).
 */
int
TseGraph::getOppositeStub(int v, int n){
	return (vertexToStub[v][n])^1;
}
/**
 * Gets index of incoming stub of edge which has outgoing end
 * as the n-th stub of source vertex v.  For directed graphs
 * this should be an odd number as it is an incoming stub.
 * Works for directed and undirected graphs.
 * 0<=n<degree(v).
 */
int
TseGraph::getOppositeInStub(int v, int n){
	return (vertexToStub[v][n])^1;
}

/**
 * Gets index of outgoing stub of edge which has incoming end
 * as the n-th stub of target vertex v.
 * This should be an even number as it is an outgoing stub.
 * No Tests done. Works for directed graphs only.
 * 0<=n<degree(v).
 */
int
TseGraph::getOppositeOutStub(int v, int n){
	return (vertexToIncomingStub[v][n])^1;
}

/**
 * Gets index of n-th neighbouring vertex of vertex v.
 * No Tests done.  If directed this assumes vertex v is a
 * source and returns the n-th neigbouring target vertex,
 * 0<=n<degree(v).
 */
int
TseGraph::getNeighbour(int v, int n){
	return stubToVertex[getOppositeStub(v, n)];
}


/**
 * Gets index of n-th neighbouring vertex of a source vertex v.
 * No Tests done. Assumes vertex v is a
 * source and returns the n-th neigbouring target vertex.  Also
 * works for undirected graphs.
 * 0<=n<outDegree(v).
 */
int
TseGraph::getNeighbouringTarget(int v, int n){
	return stubToVertex[getOppositeInStub(v, n)];
}

/**
 * Gets index of n-th neighbouring source vertex of target vertex v.
 * No Tests done. Assumes vertex v is a
 * target and returns the n-th neigbouring source vertex.  Does not
 * work for undirected graphs.
 * 0<=n<inDegree(v).
 */
int
TseGraph::getNeighbouringSource(int v, int n){
	return stubToVertex[getOppositeOutStub(v, n)];
}


/**
 * Finds the strength of a stub.
 * Works for weighted and unweighted graphs.
 * Works for directed and undirected graphs.
 */
double
TseGraph::getStubWeight(int stub){
	return (weightedGraph?edgeWeight[stub>>1]:1);
}

/*
 * Adds a new vertex with no stubs.  No checks on size performed.
 * Works for directed and undirected graphs.
 */
void
TseGraph::addVertex(){
    vertexToStub.push_back(vector<int>());
    if (directedGraph) vertexToIncomingStub.push_back(vector<int>());
}


/**
* Adds a new edge from existing vertices s and t.
* If graph is weighted then weight of 1.0 is assumed.
* No checks on vertex indices performed.
*/
void 
TseGraph::addEdge(int s, int t){
  addEdge(s,t,1.0);
}

/**
 * Adds a new edge from existing vertices s and t.
 * If graph is weighted then makes new edge of weight.
 * Checks on vertex indices and increases size if needed.
 * Works for all types of graph even unweighted (when weight is ignored).
*/
void
TseGraph::addEdgeSlow(int s, int t, double w){
    int maxSize=max(s,t)+1;
    if (vertexToStub.size()<maxSize){
                        vertexToStub.resize(maxSize);
                        if (directedGraph) vertexToIncomingStub.resize(maxSize);
  }
  addEdgeUnweighted(s,t);
  if (weightedGraph) edgeWeight.push_back(w);
}
      

/**
 * Adds a new edge from existing vertices s and t.
 * If graph is weighted then makes new edge of weight.
 * No checks on vertex indices performed.
*/
void
TseGraph::addEdge(int s, int t, double w){
  addEdgeUnweighted(s,t);
  if (weightedGraph) edgeWeight.push_back(w);
}

/**
 * Adds a new edge from existing vertices s and t.
 * Assumes graph is unweighted. 
 * No checks on vertex indices performed.
 */
void
TseGraph::addEdgeUnweighted(int s, int t){
  vertexToStub[s].push_back(stubToVertex.size());
  stubToVertex.push_back(s);
  if (directedGraph) vertexToIncomingStub[t].push_back(stubToVertex.size());
  else vertexToStub[t].push_back(stubToVertex.size());
  stubToVertex.push_back(t);
}


/**
  * Adds weight dw to edge from s to t.  Creates new edge if needed.
  * Assumes graph is unweighted.
  * No checks on vertex indices performed.
*/
void
TseGraph::addEdgeUnique(int s, int t){
	int stub=findStub(s,t);
	if (stub>=getNumberStubs() ) addEdgeUnweighted(s,t);
}
/**
  * Adds weight dw to edge from s to t.  Creates new edge if needed.
  * Assumes graph is unweighted.
  * No checks on vertex indices performed.
  */
void
TseGraph::increaseWeight(int s, int t, double dw){
	int stub=findStub(s,t);
	if (stub<getNumberStubs() ) edgeWeight[stub >>1] +=dw;
	else addEdge(s,t,dw);
}

/**
 * Finds if vertex s is connected to vertex t
 * returns stub number of s.  If this is equal to numberStubs
 * then no edge was found.  Works for directed and undirected graphs.
 * No checks on vertex indices performed.
  */
int
TseGraph::findStub(int s, int t){
        int k=vertexToStub[s].size();
	for (int n=0; n<k; n++){
		int stub = vertexToStub[s][n]  ;
		if (stubToVertex[(stub^1)]==t) return stub;
	}
	return getNumberStubs();
}
/**
 * Finds if vertex s is connected to vertex t
 * returns stub number of s.  If this is equal to numberStubs
 * then no edge was found.  Works for only for directed graphs.
 * No checks on vertex indices performed.
  */
int
TseGraph::findStubFromTarget(int s, int t){
        int k=vertexToIncomingStub[t].size();
	for (int n=0; n<k; n++){
		int stub = vertexToIncomingStub[t][n]  ;
		if (stubToVertex[(stub^1)]==s) return stub;
	}
	return getNumberStubs();
}
/**
 * Finds if vertex s is connected to vertex t
 * returns stub number of s.  If this is equal to numberStubs
 * then no edge was found.  Works for undirected and directed graphs
 * by checking the neighbours of the vertex with least number of
 * neighbours.
 * No checks on vertex indices performed.
  */
int
TseGraph::findStubFast(int s, int t){
    int ks=vertexToStub[s].size();
    if (directedGraph) {
        int kt=vertexToIncomingStub[t].size();
        if (ks<kt) return findStub(s, t);
        else return findStubFromTarget(s, t);
    }
    int kt=vertexToStub[t].size();
    if (ks<kt) return findStub(s, t);
    return findStub(t,s);
}

/**
 * Finds if vertex s is connected to vertex t.
 * Intelligently uses degrees of vertices to speed test if not directed.
 * No checks on vertex indices performed.
  */
bool
TseGraph::areConnected(int s, int t){
    int stub=-1;
    if (directedGraph || (vertexToStub[s].size()<vertexToStub[t].size()) ) stub = findStub(s,t);
    else stub = findStub(t,s);
    if (stub<getNumberStubs()) return true;
    return false;
}
/**
 * Finds if vertex s is connected to vertex t.
 * Intelligently uses degrees of vertices to speed test for both
 * directed and undirected cases.
 * No checks on vertex indices performed.
  */
bool
TseGraph::areConnectedFast(int s, int t){
    int stub=-findStubFast(s,t);
    if (stub<getNumberStubs()) return true;
    return false;
}









