/* ------------------------------------------------------------------------ 

  Header file holding definitions and declarations of variables and functions

  ------------------------------------------------------------------------*/

#include <iostream>
#include <sstream>
#include "fstream.h"
#include "iomanip.h"
#include "stdlib.h"
#include "stdio.h"
#include "math.h"
#include "complex.h"
#include "time.h"

#define PI 3.1415926535897932385

#define max(x,y) (x>=y?x:y)

/* RES = resolution; Takes values 1 = Low, 2 = Med, 3 = High
   set in routine 'getparameters' in response to user input in file 'param' */
int RES ;

/* These are the Kahler moduli in the paper hep-th/0506xxx 'a' and
   'b'. Again set in routine 'getparameters' */

double KahlerA, KahlerB ;

/* The analytic value for sqrt(det(g_{\mu\nu})) (denoted \lambda in
   the paper) and the actual value used in the Monge-Ampere equation.

   The value used in the Monge Ampere eqn, 'rootdetg' is dynamically
   determined to ensure the Monge Ampere equation converges to a
   static solution rather than slowly drifting by a constant Kahler
   transform as happens if one uses the analytic true value due to
   discretization error */

double analrootdetg, rootdetg ;

/* These store the cutoffs for the coordinate patches.

The K1 EH patch has;    sigma<MAXSIG, |y|<MAXY

The K2 torus patch has; sigma>MINSIG

(K3 EH patch has sigma'=sigma<MAXSIG, |y'|<MAXY)

Hence the patches overlap.

The value MINSIG < TRANSSIG < MAXSIG is just some intermediate value
used occasionally in the code. This is the value where if we present
results, for sigma less than TRANSSIG we use the EH patch value and
for greater the torus value.

*/

double MAXY, MINSIG, TRANSSIG, MAXSIG ;

/* Sets degree of interpolation when getting Kahler potential from
   neighbouring patches. For lowest resolution (RES=1) uses 1st order
   interpolation (INTERP=1) as there aren't enough points on the patch
   overlaps to do higher order interpolation. For medium and higher
   resolutions (RES>=2) we use 3rd order interpolation (INTERP=3) */

int     INTERP ;

/* 

Number of points in Re(w),Im(w) lattice directions = NW1+4
                    Re(y),Im(y)                    = NY1+4
                    Re(z1,z2),Im(z1,z2)            = NZ2+4

The '+4's are to include some additional buffer points outside the
'interior' points of the patches, which should be thought of as being
in the neighbouring patches and whole values are filled by
interpolating from these patches. These points are then accessed when
one evaluates the Monge-Ampere equation at an interior point right at
the edge of the interior of one of the patches.

*/

int     NW1, NY1, NZ ;
double  DW1, DY1, DZ ;

/* Pointers to the K1,K2 arrays holding the Kahler potentials in the
   EH and torus patches, as well as the templates tempK1,tempK2 which
   store which points are in the interior of the patches (value 0 in
   template array) and are relaxed, which are in the buffer zones just
   outside the interior (value 1 in array) and are needed to evaluate
   the Monge Ampere equation at the edge of the interior of the patch,
   and so are interpolated from the neighbouring patches, and finally
   points which are just simply outside the patch (ie. in K1 array,
   points with sigma>MAXSIG) */ 

double *K1,     *K2     ;
char   *tempK1, *tempK2 ;

/* Whilst the K1 array can be thought of as being rectangular, the K2
   array is more complicated as we build in the discrete symmetries
   K2(z1,z2) = K2(z2,z1) and K2(z1,z2) = K2(\bar{z1},\bar{z2}). Hence
   K2pos is an indexing array used to find a point in K2. See K3lat.c
   routine 'initializememory' for usage and initialization. K2size is
   also used for similar purposes */

long   *K2pos ;
long    K2size ;


/* Again pos1 and pos2 are compiler macros to simplify accessing the
   K1 and K2 arrays. The long integers are also used for similar
   purposes. see comments in 'initializememory' function for
   details. The long intergers are set up when the resolution is
   initialized in the 'getparmeters' routine */

#define pos1(aa,bb,cc,dd)  ((aa)*K1A + (bb)*K1B + (cc)*K1C + (dd))
#define pos2(aa,bb,cc)  ((aa)*K2A + (bb)*K2B + (cc))

long K1dimA, K1dimB, K1dimC, K1dimD ;
long K2dimA, K2dimB, K2dimC, K2dimD ;

long K1A, K1B, K1C ;
long K2A, K2B, K2C ;


/* An array used in routine `UpdateK2_2nd' */

double B[5][5][5][5] ;



/* Declaration of functions used in K3.c and K3lat.c */

void getparameters(void) ;

void initializememory(void) ;

void initK1(void) ;
void initK2(void) ;

complex<double> coord_w1(int,int), coord_y1(int,int) ;
complex<double> lat_w1(complex<double>), lat_y1(complex<double>) ;

complex<double> coord_z1(int,int), coord_z2(int,int) ;
complex<double> lat_z1(complex<double>), lat_z2(complex<double>) ;

double getK1(int,int,int,int), getK1fromK3(int,int,int,int) ;
double getK3(int,int,int,int), getK3fromK1(int,int,int,int) ;
double getK2(int,int,int,int) ;

double getK1fromK2(int,int,int,int) ;
double getK3fromK2(int,int,int,int) ;
double getK2fromK1(int,int,int,int) ;
double getK2fromK3(int,int,int,int) ;

double UpdateK1_2nd(int,int,int,int,double*) ;
double UpdateK2_2nd(int,int,int,int,double*) ;


double relax(bool) ;
void calcanaldetg(void) ; 

void initinfo(void) ;

void savebin(void) ;

void inittemp(void) ;
void fill(void) ;

