

/* This routine allocates the memory for the K1 and K2 arrays (and
   also the template arrays tempK1, tempK2 which store whether points
   are actually inside the coordinate patches, or are at the edges and
   hence must be interpolated from the neighbouring patches). This is
   done dynamically as the memory requried can be large.

   It also initializes another array K2pos which is used to get K2
   values from the K2 array in such a way that the K2 array builds in
   the addition 3 discrete symmetries outlined at the top of the K3.c
   file. This is a bit fiddly but saves approx a factor of 8 in
   memory!

 */


void initializememory(void) 
{
  int z1re, z1im, z2re, z2im ;
  long count ;


  /* Dynamically allocate a rectangular array of the required size for K1 */

  cout << "Allocating K1... " ; cout.flush() ;

  K1 = (double *) malloc(K1dimA*K1dimB*K1dimC*K1dimD*sizeof(double)) ;
  if(K1==NULL) { cout << "K1 Allocation error" << endl ; exit(1) ; }

  tempK1 = (char *) malloc(K1dimA*K1dimB*K1dimC*K1dimD*sizeof(char)) ;
  if(tempK1==NULL) { cout << "tempK1 Allocation error" << endl ; exit(1) ; }

  cout << "Done" << endl ;


  /* We will allocate a linear array for K2 and then use the array
     K2pos to translate the natural rectangular labeling in terms of
     real, imag parts of z1 and z2 to the linear position in the
     actual K2 array */

  cout << "Allocating K2... " ; cout.flush() ;

  /* First compute the size required to store K2 taking into account
     the 3 discrete symmetries 

     Note: in the K2 patch we still allocate space for the sigma <
     MINSIG points which is a bit wasteful but not very inefficient

  */


  K2size = 0 ;

  /* Loop over all points potentially in K2 patch */
  
  for(z1re=0; z1re<NZ+4; z1re++) {
    for(z1im=0; z1im<NZ+4; z1im++) {
      for(z2re=0; z2re<NZ+4; z2re++) {
	
	/* Last line in loop (partially) builds in the 3 discrete
	   symmetries. We use the function `reflectZ' to map a point
	   with z2im not in this range into this fundamental range stored */
	
	for(z2im=max(z1re,z1im); z2im<NZ+4; z2im++) {
		  
	  K2size++ ;
	  
	}
      }   
    }
  }

  /* K2size now holds the number of points required for K2 array, so
     allocate this memory */

  cout << "Allocating K2!" << endl ;

  K2 = (double *) malloc(K2size*sizeof(double)) ;
  if(K2==NULL) { cout << "K2 Allocation error" << endl ; exit(1) ; }

  tempK2 = (char *) malloc(K2size*sizeof(char)) ;
  if(tempK2==NULL) { cout << "tempK2 Allocation error" << endl ; exit(1) ; }

  /* Allocate the indexing array K2pos */

  K2pos = (long *) malloc(K2dimA*K2dimB*K2dimC*sizeof(long)) ;
  if(K2pos==NULL) { cout << "K2 position Allocation error" << endl ; exit(1) ; }
  
  /* Now set up the indexing array K2pos */

  count = 0 ;

  for(z1re=0; z1re<NZ+4; z1re++) {
    for(z1im=0; z1im<NZ+4; z1im++) {  
      for(z2re=0; z2re<NZ+4; z2re++) {
	
	K2pos[pos2(z1re,z1im,z2re)] = count - max(z1re,z1im) ;

	for(z2im=max(z1re,z1im); z2im<NZ+4; z2im++) {

	  count++ ;
	  
	}
	
      }
    }
  }

  /* The syntax for accessing points in K1 and K2 is now:


  K1[pos1(wre,wim,yre,yim)]

  and 

  K2[K2pos[pos2(z1re,z1im,z2re)]+z2im]


  where pos1 and pos2 are just #defined in the header file as;

  
  #define pos1(aa,bb,cc,dd)  ((aa)*K1A + (bb)*K1B + (cc)*K1C + (dd))

  #define pos2(aa,bb,cc)     ((aa)*K2A + (bb)*K2B + (cc))


  and the long integers K1A,B,C, K2A,B are defined in the routine
  'getparameters' in the main file as;

  K1A = (K1dimB*K1dimC*K1dimD) ;
  K1B =        (K1dimC*K1dimD) ;
  K1C =               (K1dimD) ;
  
  K2A = (K2dimB*K2dimC) ;
  K2B =        (K2dimC) ;

  */

}


/* Routines to map points that are outside the fundamental domain of
   the K1 EH patch, but that will be required to evaluate the
   Monge-Ampere equation within - but at the edge of - the fundamental
   domain, back into it using the discrete symmetries.

   The two routines fboundaryWY and iboundaryWY are the same, just
   taking doubles or integers for the lattice point locations.

   Fundamental domains for K1 are Re(w), Im(w), Re(y), Im(y) > 0
   Patch then extends out to sigma = MAXSIG

*/


void fboundaryWY(double *wre, double *wim, double *yre, double *yim)
{
  double tmp ;

  /* If Re(y)<0 and Im(y)<0 use symmetry K(w,y) = K(w,-y) to map back
     to fundamental domain with Re(y), Im(y) > 0*/

  if(*yre<2. && *yim<2.) {

    *yre = 4. - *yre ;
    *yim = 4. - *yim ;

  }

  /* If Re(y)<0 use symmetry K(w,y) = K(w,-i y) */

  if(*yre<2.) {

    tmp = *yim ;

    *yim = 4. - *yre ;
    *yre =    tmp ;

  }

  /* If Im(y)<0 use symmetry K(w,y) = K(w,+i y) */

  if(*yim<2.) {

    tmp = *yre ;

    *yre = 4. - *yim ;
    *yim =    tmp ;

  }

  /* If Re(w)<0 use symmetry K(w,y) = K(+i \bar{w},\bar{y}) */

  if(*wre<2.) {

    *wre = 4. - *wre ;

    tmp = *yre ;

    *yre = *yim ;

    *yim = tmp ;

  }

  /* If Im(w)<0 use symmetry K(w,y) = K(-i \bar{w},\bar{y}) */

  if(*wim<2.) {

    *wim = 4. - *wim ;

    tmp = *yre ;

    *yre = *yim ;

    *yim = tmp ;

  }

}

/* Same as above but with integers */

void iboundaryWY(int *wre, int *wim, int *yre, int *yim)
{
  int tmp ;

  if(*yre<2 && *yim<2) {

    *yre = 4 - *yre ;
    *yim = 4 - *yim ;

  }

  if(*yre<2) {

    tmp = *yim ;

    *yim = 4 - *yre ;
    *yre =    tmp ;

  }

  if(*yim<2) {

    tmp = *yre ;

    *yre = 4 - *yim ;
    *yim =    tmp ;

  }

  if(*wre<2) {

    *wre = 4 - *wre ;

    tmp = *yre ;

    *yre = *yim ;

    *yim = tmp ;

  }

  if(*wim<2) {

    *wim = 4 - *wim ;

    tmp = *yre ;

    *yre = *yim ;

    *yim = tmp ;

  }

}



/* 

   Fundamental domains for K2 are 0.25 > Re(z1), Im(z1), Re(z2), Im(z2) > 0
   Patch then extends in to sigma = MINSIG

   The discrete symmetry additionally allows the fundamental domain to
   further be reduced so that;

   Im(z2) > max( Re(z1), Im(z1) )

*/


/*
  
   The following 2 routines deal with points that have one (or more)
   of Re(z1), Im(z1), Re(z2), Im(z2) < 0

   Since the discrete symmetry K(z1,z2) = K(\pm i z1,z2) = K(z1,\pm i
   z2), we only need one routine for a general coordinate z, which
   could be either z1 or z2

   These routines map coordinate z back into the fundamental domain if
   Re(z) or Im(z) < 0

*/

void fboundaryZ(double *re, double *im)
{
  double tmp ;

  /* If Re(z) and Im(z) < 0 use symmetry in z; f(z) = f(-z) */

  if(*re<2. && *im<2.) {

    *re = 4. - *re ;
    *im = 4. - *im ;

  }

  /* If Re(z) < 0 use symmetry in z; f(z) = f(-i z) */

  if(*re<2.) {

    tmp = *im ;

    *im = 4. - *re ;
    *re =      tmp ;
   
  }

  /* If Im(z) < 0 use symmetry in z; f(z) = f(+i z) */

  if(*im<2.) {
    
    tmp = *re ;

    *re = 4. - *im ;
    *im =      tmp ;

  }

}

void iboundaryZ(int *re, int *im)
{
  int tmp ;

  if(*re<2 && *im<2) {

    *re = 4 - *re ;
    *im = 4 - *im ;

  }

  if(*re<2) {

    tmp = *im ;

    *im = 4 - *re ;
    *re =     tmp ;

  }

  if(*im<2) {

    tmp = *re ;

    *re = 4 - *im ;
    *im =     tmp ;

  }

}


/* 

The following routine deals with the situation Im(z2) <= max( Re(z1), Im(z1) )
and uses the discrete symmetries;

K(z1,z2) = K(z2,z1) = K(\bar{z1},\bar{z2})

to map the point back into the fundamental domain

*/


void reflectZ(int *z1re, int *z1im, int *z2re, int *z2im)
{
  int tmp ;

  if(*z2im>=max(*z1re,*z1im)) {

    /* if already in fundamental domain don't do anything */

    return ;

  } 
  /* so if you reach here you are *not* in the fundamental domain
     Im(z2) > max( Re(z1), Im(z1) ) */
  else if(*z2re>=max(*z1re,*z1im)) {

    /* if Re(z2)>max(Re(z1),Im(z1) then by using 

       K(z1,z2) = K(\bar{z1},\bar{z2}) 

       you can map back to the fundamental domain with 
       Im(z2) > max( Re(z1), Im(z1) ) */

    tmp  = *z1re ; *z1re = *z1im ; *z1im = tmp ;
    tmp  = *z2re ; *z2re = *z2im ; *z2im = tmp ;

    return ;

  } else if(*z1im>=max(*z2re,*z2im)) {

    /* if Im(z1)>max(Re(z2),Im(z2) then by using 

       K(z1,z2) = K(z2,z1) 

       you can map back to the fundamental domain with 
       Im(z2) > max( Re(z1), Im(z1) ) */

    tmp  = *z2re ; *z2re = *z1re ; *z1re = tmp ;
    tmp  = *z2im ; *z2im = *z1im ; *z1im = tmp ;

    return ;

  } else  if(*z1re>=max(*z2re,*z2im)) {

    /* The last option, if the above conditions are not met is then
       that Re(z1)>max(Re(z2),Im(z2))

       then by using 

       K(z1,z2) = K(\bar{z2},\bar{z1}) 

       you can map back to the fundamental domain with 
       Im(z2) > max( Re(z1), Im(z1) ) */

    tmp  = *z2re ; *z2re = *z1im ; *z1im = tmp ;
    tmp  = *z2im ; *z2im = *z1re ; *z1re = tmp ;

    return ;

  } else {

    cout << "Logical impossibility!" << endl ; exit(1) ;

  }

}



/* The following routines coord* convert the integer lattice positions
   for the real and imag parts of the discretized coordinates for w, y
   and z1,z2 into the actual complex coordinate values.

   Likewise the routines lat* map the complex coordinates into the
   real and imaginary lattice points, although they do not actually
   convert back to integers. These routines are used by the
   interpolation routines later...

 */

complex<double> coord_w1(int w1re, int w1im)
{

  w1re -= 2 ; w1im -= 2 ;

  return complex<double>( ( w1re )*DW1, (w1im )*DW1 ) ;

}

complex<double> coord_y1(int y1re, int y1im)
{

  y1re -= 2 ; y1im -= 2 ;

  return complex<double>( (y1re )*DY1, (y1im )*DY1 ) ;

}

complex<double> lat_w1(complex<double> w1,complex<double> y1)
{
  double w1re, w1im, y1re, y1im ;

  w1re = w1.real()/DW1 + 2. ; w1im = w1.imag()/DW1 + 2. ;

  y1re = y1.real()/DY1 + 2. ; y1im = y1.imag()/DY1 + 2. ;

  fboundaryWY(&w1re,&w1im,&y1re,&y1im) ;

  return complex<double>( w1re , w1im ) ;

}

complex<double> lat_y1(complex<double> w1,complex<double> y1)
{
  double w1re, w1im, y1re, y1im ;

  w1re = w1.real()/DW1 + 2. ; w1im = w1.imag()/DW1 + 2. ;

  y1re = y1.real()/DY1 + 2. ; y1im = y1.imag()/DY1 + 2. ;

  fboundaryWY(&w1re,&w1im,&y1re,&y1im) ;

  return complex<double>( y1re , y1im ) ;

}


complex<double> coord_z1(int z1re, int z1im)
{

  return complex<double>( (z1re - 2.)*DZ, (z1im - 2.)*DZ ) ;

}

complex<double> coord_z2(int z2re, int z2im)
{

  return complex<double>( (z2re - 2.)*DZ, (z2im - 2.)*DZ ) ;

}

complex<double> lat_z1(complex<double> z1)
{
  double tmp, z1re, z1im ;

  z1re = z1.real()/DZ + 2. ; z1im = z1.imag()/DZ + 2. ;

  fboundaryZ(&z1re,&z1im) ;

  return complex<double>( z1re , z1im ) ;

}

complex<double> lat_z2(complex<double> z2)
{
  double tmp, z2re, z2im ;

  z2re = z2.real()/DZ + 2. ; z2im = z2.imag()/DZ + 2. ;

  fboundaryZ(&z2re,&z2im) ;

  return complex<double>( z2re , z2im ) ;

}




/* The routines getK1, getK2 are the two key routines. They take as
   arguement the integer lattice position w,y or z1,z2. If this
   position is within the fundamental domain they return the value
   from the K1 or K2 array.

   However, if the value is outside the coordinate patch as sigma is
   too large (for K1) or too small (for K2), or for K1 |y| is too
   large, they call the routines to interpolate the values from the
   other patches.

   These interpolation functions (see later) then call these getK1,2
   routines at various points in the other patch and use this info to
   interpolate a value using 3rd order interpolation for the medium
   and high resolution and only 1st order for low (since at this low
   resolution the patches don't overlap enough to do higher order
   interpolation).

   Note that since the interpolation routines also call these getK1,2
   functions then if these points required for interpolation are also
   outside the relevent patches then they too are then interpolated
   and so on. The process is recursive.

   It will only give a result if the discretized coordinate patch
   geometry has sufficient overlap to allow for the 3rd order
   interpolation, and doesn't have too much overlap so that points
   transformed into other patches still lie inside but near the
   boundaries of those patches.

   The patch geometries set up in routine 'getparameter' are designed
   to give good performance for these interpolations, requiring little
   recursion.

   These interpolation routines are slow, and this is probably the
   first thing one should improve if modifying the code. However they
   work fine.

*/


/* General function to be called to get K1 value at lattice point if
   don't know whether point is in the interior of the EH patch */

double getK1(int wrein, int wimin, int yrein, int yimin)
{
  int tmp, w1re, w1im, y1re, y1im ;

  double sig, absy ;

  w1re = wrein ; w1im = wimin ; y1re = yrein ; y1im = yimin ;

  /* If Re(w),Im(w),Re(y) or Im(y) < 0, the point is outside the
     fundamental domain, and is mapped back using ... */

  iboundaryWY(&w1re,&w1im,&y1re,&y1im) ;
  
  /* Is the point inside sigma<MAXSIG and |y|<ymax? */

  sig  = 2.*abs(coord_w1(w1re,w1im))*(1.+abs(coord_y1(y1re,y1im))*abs(coord_y1(y1re,y1im))) ;
  
  absy = abs(coord_y1(y1re,y1im)) ;


  if(sig<=MAXSIG) { 
    
    if(absy<=MAXY) {

      /* The point is now inside the patch and the potential can just
	 be got from the K1 array */

      return( K1[pos1(w1re,w1im,y1re,y1im)] ) ;
      
    } else {

      /* The point has |y|>ymax but sigma<=MAXSIG so lies in the other
	 EH patch (w',y' coordinates).

	 One might just interpolate direct from this other EH patch
	 'K3' although K3 is identical to K1 up to a Kahler transform.

	 However, if sigma < transsig (which is intermediate between
	 minsig and maxsig) it is better to get the value from the
	 torus patch as the transform used to map to the other EH patch;

	 w' = w y^2

	 y' = 1/y

	 increases sigma' in the new patch (as |y|>1), so if sigma is
	 already pretty large, then when we map to the other EH patch
	 'K3' then sigma there will probably be larger than MAXSIG so
	 we will just have to go to the torus patch at that stage.

      */

      if(sig<=TRANSSIG) 

	return( getK1fromK3(w1re,w1im,y1re,y1im) ) ;
      
      else 

	return( getK1fromK2(w1re,w1im,y1re,y1im) ) ;

    }

  } else {

      /* The point is outside the patch as sigma>MAXSIG

      We therefore interpolate the potential from the torus patch

      (Note that |y| could also be > ymax but that doesn't change the
      fact we must interpolate from the torus patch)

      */

      return( getK1fromK2(w1re,w1im,y1re,y1im) ) ;

  }

}


/* This routine interpolates the Kahler potential in the EH patch K1
   from the other EH patch 'K3' which is actually the same as K1 up to
   a (Kahler modulus 'a' dependent) Kahler transformation
*/

double getK1fromK3(int w1re, int w1im, int y1re, int y1im)
{
  complex<double> w1, y1, w1dash, y1dash, latw, laty ;
  
  int wdashre, wdashim, ydashre, ydashim ;
  int tmpwre, tmpwim, tmpyre, tmpyim ;  

  int i1,i2,i3,i4 ;
  double val, val2, oldval ;

  double x, frc[4], mult[4][4] ;



  /* Convert w,y lattice point to complex w,y coordinates */

  w1 = coord_w1(w1re,w1im) ;

  y1 = coord_y1(y1re,y1im) ;

  /* Transform these to the w', y' coordinates */

  y1dash = ( complex<double>(1.,0) ) / y1 ;

  w1dash = w1*y1*y1 ;

  /* Remember that the potential K(w,y) = K(w',y') + kahler transform! 

  Hence we will get the potential from K1 at the position w = w', y =
  y' and then add this transform
  
  */

  /* Compute the lattice position of w',y' - remember these variables
     latw, laty are not yet converted to integers as they will be
     fractional - ie. the lattice points in one patch do not transform
     to the lattice points in another */

  latw = lat_w1(w1dash,y1dash) ;
  laty = lat_y1(w1dash,y1dash) ;

  /* Convert these to integers ... */

  wdashre = (int) floor(latw.real()) ;
  wdashim = (int) floor(latw.imag()) ;

  ydashre = (int) floor(laty.real()) ;
  ydashim = (int) floor(laty.imag()) ;

  /* ... but keep track of their fractional parts as these are needed
     for the interpolation */

  frc[0] = latw.real() - ((double) wdashre) ;
  frc[1] = latw.imag() - ((double) wdashim) ;

  frc[2] = laty.real() - ((double) ydashre) ;
  frc[3] = laty.imag() - ((double) ydashim) ;



  /* The interpolation is computed as;

  K(i,j,k,l) = sum_{a,b,c,d = -1 to 2} mult(0,a)*mult(1,b)*mult(2,c)*mult(3,d)*K(i+a,j+b,k+c,l+d)

  where mult(0...3,n) are computed approriately in terms of the
  fractional parts of the desired position of the lattice point
  (ie. frc[0...3]) to give 1st (lowest resolution) or 3rd (medium and
  higher resolutions) interpolation
  
  */

  /* Here mult is computed for either 1st or 3rd order interpolation */

  for(i1=0;i1<4;i1++) {
 
    x = frc[i1] ;

    switch(INTERP) {

    case(1) :
      /* 1st Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = 1.-x ;
      mult[i1][2] = x ;
      mult[i1][3] = 0 ;
      break ;

    case(3) :
      /* 3rd Order */
      mult[i1][0] = -x*(x-1.)*(x-2.)/6. ;
      mult[i1][1] = ( x*x*x - 2.*x*x - x + 2. )/2. ;
      mult[i1][2] = x*( -x*x + x + 2. )/2. ;
      mult[i1][3] = x*(x*x-1.)/6. ;
      break ;

    default : 
      cout << "Error in INTERP (" << INTERP << ")" << endl ; 
      exit(1) ;

    }

  }

  /* interpolated value will be stored in val */

  val = 0. ;

  /* Now compute the sum:
     sum_{a,b,c,d = -1 to 2} mult(0,a)*mult(1,b)*mult(2,c)*mult(3,d)*K(i+a,j+b,k+c,l+d) */


  for(i1=-1;i1<=2;i1++) {
    for(i2=-1;i2<=2;i2++) {
      for(i3=-1;i3<=2;i3++) {
	for(i4=-1;i4<=2;i4++) {

	  val2 = 1. ;
	  
	  switch(i1) {
	    
	  case(-1) : val2 *= mult[0][0] ; break ;
	    
	  case(0) : val2 *= mult[0][1] ; break ;
	    
	  case(1) : val2 *= mult[0][2] ; break ;
	    
	  case(2) : val2 *= mult[0][3] ; break ;

	    /* Just for fun! Should never get here! */

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i2) {
	    
	  case(-1) : val2 *= mult[1][0] ; break ;
	    
	  case(0) : val2 *= mult[1][1] ; break ;
	    
	  case(1) : val2 *= mult[1][2] ; break ;
	    
	  case(2) : val2 *= mult[1][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i3) {
	    
	  case(-1) : val2 *= mult[2][0] ; break ;
	    
	  case(0) : val2 *= mult[2][1] ; break ;
	    
	  case(1) : val2 *= mult[2][2] ; break ;
	    
	  case(2) : val2 *= mult[2][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i4) {
	    
	  case(-1) : val2 *= mult[3][0] ; break ;
	    
	  case(0) : val2 *= mult[3][1] ; break ;
	    
	  case(1) : val2 *= mult[3][2] ; break ;
	    
	  case(2) : val2 *= mult[3][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  /* If val2 = 0 which could happen even for 3rd order
	     interpolation if one of the fractional lattice positions
	     vanished, (ie. the lattice position actually did fall on
	     a lattice point position in one or more of the coordinate
	     directions) then don't bother getting the value as it
	     will be zero in the sum.  */

	  if(val2!=0.) {

	    tmpwre = wdashre+i1 ; tmpwim = wdashim+i2 ; tmpyre = ydashre+i3 ; tmpyim = ydashim+i4 ;

	    /* In the case the interpolation point is outside the
	       domain with Re(w), Im(w), Re(y) or Im(y) < 0 then use
	       iboundaryWY to map it back in */

	    iboundaryWY(&tmpwre,&tmpwim,&tmpyre,&tmpyim) ;

	    /* Now if the point in the sum is an interior point, we
	       can just compute the potential by getting its value
	       from the K1 array. If not, we simply recursively call
	       getK3 (ie. get a point in the w',y' EH patch) knowing
	       that this will have to go away and interpolate this
	       point. Providing the coordinate patches are designed
	       correctly (as they are as defined in 'getparameter'
	       routine), this recursion stops after a finite (and
	       hopefully small) number of iterations. */

	    if( tempK1[pos1(tmpwre,tmpwim,tmpyre,tmpyim)]==0 ) 
	      val += val2*K1[pos1(tmpwre,tmpwim,tmpyre,tmpyim)] ;
	    else
	      val += val2*getK3(tmpwre,tmpwim,tmpyre,tmpyim) ;
	  
	  }

	}
      }
    }
  }

  /* Now we have computed the sum that interpolates the potential to
     either 1st or 3rd order using the neighbouring lattice sites, we
     must add the correct kahler potential to correctly yield the
     value in the K1 EH w,y coordinate patch */

  val += 0.5*pow(KahlerA,2)*log(abs(y1)*abs(y1)) ;

  return( val ) ;

}


/* This routine does exactly the same thing, but instead of
   interpolating K1 from K3 (the w',y' EH coordinate patch), it
   interpolates it from K2. */

double getK1fromK2(int w1re, int w1im, int y1re, int y1im)
{
  complex<double> z1, z2, w, y, latz1, latz2 ;
  
  int z1dashre, z1dashim, z2dashre, z2dashim ;
  int tmpz1re, tmpz1im, tmpz2re, tmpz2im ;
   
  int i1, i2, i3, i4 ;
  double val, val2 ;

  double x, frc[4], mult[4][4] ;


  w = coord_w1(w1re,w1im) ;  y = coord_y1(y1re,y1im) ;

  z1 = sqrt(2.*w) ;
  z2 = sqrt(2.*w*y*y) ;

  latz1 = lat_z1(z1) ;
  latz2 = lat_z2(z2) ;

  z1dashre = (int) floor(latz1.real()) ;
  z1dashim = (int) floor(latz1.imag()) ;

  z2dashre = (int) floor(latz2.real()) ;
  z2dashim = (int) floor(latz2.imag()) ;

  frc[0] = latz1.real() - ((double) z1dashre) ;
  frc[1] = latz1.imag() - ((double) z1dashim) ;

  frc[2] = latz2.real() - ((double) z2dashre) ;
  frc[3] = latz2.imag() - ((double) z2dashim) ;

  for(i1=0;i1<4;i1++) {
 
    x = frc[i1] ;

    switch(INTERP) {

    case(1) :
      /* 1st Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = 1.-x ;
      mult[i1][2] = x ;
      mult[i1][3] = 0 ;
      break ;

    case(2) :
      /* 2nd Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = (x-1.)*(x-2.)/2. ;
      mult[i1][2] = -x*(x-2.) ;
      mult[i1][3] = x*(x-1)/2. ;
      break ;

    case(3) :
      /* 3rd Order */
      mult[i1][0] = -x*(x-1.)*(x-2.)/6. ;
      mult[i1][1] = ( x*x*x - 2.*x*x - x + 2. )/2. ;
      mult[i1][2] = x*( -x*x + x + 2. )/2. ;
      mult[i1][3] = x*(x*x-1.)/6. ;
      break ;

    default : 
      cout << "Error in INTERP (" << INTERP << ")" << endl ; 
      exit(1) ;

    }

  }

  val = 0. ;

  for(i1=-1;i1<=2;i1++) {
    for(i2=-1;i2<=2;i2++) {
      for(i3=-1;i3<=2;i3++) {
	for(i4=-1;i4<=2;i4++) {

	  val2 = 1. ;
	  
	  switch(i1) {
	    
	  case(-1) : val2 *= mult[0][0] ; break ;
	    
	  case(0) : val2 *= mult[0][1] ; break ;
	    
	  case(1) : val2 *= mult[0][2] ; break ;
	    
	  case(2) : val2 *= mult[0][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i2) {
	    
	  case(-1) : val2 *= mult[1][0] ; break ;
	    
	  case(0) : val2 *= mult[1][1] ; break ;
	    
	  case(1) : val2 *= mult[1][2] ; break ;
	    
	  case(2) : val2 *= mult[1][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i3) {
	    
	  case(-1) : val2 *= mult[2][0] ; break ;
	    
	  case(0) : val2 *= mult[2][1] ; break ;
	    
	  case(1) : val2 *= mult[2][2] ; break ;
	    
	  case(2) : val2 *= mult[2][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i4) {
	    
	  case(-1) : val2 *= mult[3][0] ; break ;
	    
	  case(0) : val2 *= mult[3][1] ; break ;
	    
	  case(1) : val2 *= mult[3][2] ; break ;
	    
	  case(2) : val2 *= mult[3][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }


	  tmpz1re = z1dashre+i1 ;
	  tmpz1im = z1dashim+i2 ;

	  tmpz2re = z2dashre+i3 ;
	  tmpz2im = z2dashim+i4 ;

	  iboundaryZ(&tmpz1re,&tmpz1im) ;
	  iboundaryZ(&tmpz2re,&tmpz2im) ;
	  reflectZ(&tmpz1re,&tmpz1im,&tmpz2re,&tmpz2im) ;

	  if(val2!=0.) {
	    val += val2*K2[K2pos[pos2(tmpz1re,tmpz1im,tmpz2re)]+tmpz2im] ;
	    if(tempK2[K2pos[pos2(tmpz1re,tmpz1im,tmpz2re)]+tmpz2im]==-1) {
	      cout << "Terrible error!" << endl ;
	    }
	  }

	}
      }
    }
  }
  
  val -= 0.5*pow(KahlerA,2)*log(abs(z1)*abs(z1)) ;

  return( val ) ;


}



/* This routine is the same as the getK1 routine, since K1(w,y) = K3(w,y) due to the discrete symmetry */


double getK3(int wrein, int wimin, int yrein, int yimin)
{
  int tmp, w1re, w1im, y1re, y1im ;

  double sig, absy ;

  w1re = wrein ; w1im = wimin ; y1re = yrein ; y1im = yimin ;

  iboundaryWY(&w1re,&w1im,&y1re,&y1im) ;


  sig  = 2.*abs(coord_w1(w1re,w1im))*(1.+abs(coord_y1(y1re,y1im))*abs(coord_y1(y1re,y1im))) ;
  
  absy = abs(coord_y1(y1re,y1im)) ;


  if(sig<=MAXSIG) { 
    
    if(absy<=MAXY) {

      return( K1[pos1(w1re,w1im,y1re,y1im)] ) ;
      
    } else {

      /* Possible triple overlap */

      if(sig<=TRANSSIG) 

	return( getK3fromK1(w1re,w1im,y1re,y1im) ) ;
      
      else 

	return( getK3fromK2(w1re,w1im,y1re,y1im) ) ;

    }

  } else {

      return( getK3fromK2(w1re,w1im,y1re,y1im) ) ;

  }

}


/* Same as getK1fromK3 but with different Kahler transform */

double getK3fromK1(int w1re, int w1im, int y1re, int y1im)
{
  complex<double> w1, y1, w1dash, y1dash, latw, laty ;
  
  int wdashre, wdashim, ydashre, ydashim ;
  int tmpwre, tmpwim, tmpyre, tmpyim ;    

  int i1,i2,i3,i4 ;
  double val, val2, oldval ;

  double x, frc[4], mult[4][4] ;


  w1 = coord_w1(w1re,w1im) ;

  y1 = coord_y1(y1re,y1im) ;

  y1dash = ( complex<double>(1.,0) ) / y1 ;

  w1dash = w1*y1*y1 ;

  latw = lat_w1(w1dash,y1dash) ;
  laty = lat_y1(w1dash,y1dash) ;

  wdashre = (int) floor(latw.real()) ;
  wdashim = (int) floor(latw.imag()) ;

  ydashre = (int) floor(laty.real()) ;
  ydashim = (int) floor(laty.imag()) ;

  frc[0] = latw.real() - ((double) wdashre) ;
  frc[1] = latw.imag() - ((double) wdashim) ;

  frc[2] = laty.real() - ((double) ydashre) ;
  frc[3] = laty.imag() - ((double) ydashim) ;

  for(i1=0;i1<4;i1++) {
 
    x = frc[i1] ;

    switch(INTERP) {

    case(1) :
      /* 1st Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = 1.-x ;
      mult[i1][2] = x ;
      mult[i1][3] = 0 ;
      break ;
      
    case(2) : 
      /* 2nd Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = (x-1.)*(x-2.)/2. ;
      mult[i1][2] = -x*(x-2.) ;
      mult[i1][3] = x*(x-1)/2. ;
      break ;

    case(3) :
      /* 3rd Order */
      mult[i1][0] = -x*(x-1.)*(x-2.)/6. ;
      mult[i1][1] = ( x*x*x - 2.*x*x - x + 2. )/2. ;
      mult[i1][2] = x*( -x*x + x + 2. )/2. ;
      mult[i1][3] = x*(x*x-1.)/6. ;
      break ;

    default : 
      cout << "Error in INTERP (" << INTERP << ")" << endl ; 
      exit(1) ;

    }

  }

  val = 0. ;

  for(i1=-1;i1<=2;i1++) {
    for(i2=-1;i2<=2;i2++) {
      for(i3=-1;i3<=2;i3++) {
	for(i4=-1;i4<=2;i4++) {

	  val2 = 1. ;
	  
	  switch(i1) {
	    
	  case(-1) : val2 *= mult[0][0] ; break ;
	    
	  case(0) : val2 *= mult[0][1] ; break ;
	    
	  case(1) : val2 *= mult[0][2] ; break ;
	    
	  case(2) : val2 *= mult[0][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i2) {
	    
	  case(-1) : val2 *= mult[1][0] ; break ;
	    
	  case(0) : val2 *= mult[1][1] ; break ;
	    
	  case(1) : val2 *= mult[1][2] ; break ;
	    
	  case(2) : val2 *= mult[1][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i3) {
	    
	  case(-1) : val2 *= mult[2][0] ; break ;
	    
	  case(0) : val2 *= mult[2][1] ; break ;
	    
	  case(1) : val2 *= mult[2][2] ; break ;
	    
	  case(2) : val2 *= mult[2][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i4) {
	    
	  case(-1) : val2 *= mult[3][0] ; break ;
	    
	  case(0) : val2 *= mult[3][1] ; break ;
	    
	  case(1) : val2 *= mult[3][2] ; break ;
	    
	  case(2) : val2 *= mult[3][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  if(val2!=0.) {
	    
	    tmpwre = wdashre+i1 ; tmpwim = wdashim+i2 ; tmpyre = ydashre+i3 ; tmpyim = ydashim+i4 ;
	    
	    iboundaryWY(&tmpwre,&tmpwim,&tmpyre,&tmpyim) ;
	    
	    if( tempK1[pos1(tmpwre,tmpwim,tmpyre,tmpyim)]==0 ) 
	      val += val2*K1[pos1(tmpwre,tmpwim,tmpyre,tmpyim)] ;
	    else
	      val += val2*getK1(tmpwre,tmpwim,tmpyre,tmpyim) ;

	  }

	}
      }
    }
  }

  val += 0.5*pow(KahlerA,2)*log(abs(y1)*abs(y1)) ;

  return( val ) ;

}


/* Same as getK1fromK2 but with different Kahler transform */

double getK3fromK2(int w1re, int w1im, int y1re, int y1im)
{
  complex<double> z1, z2, w, y, latz1, latz2 ;
  
  int z1dashre, z1dashim, z2dashre, z2dashim ;
  int tmpz1re, tmpz1im, tmpz2re, tmpz2im ;

  int i1, i2, i3, i4 ;
  double val, val2 ;

  double x, frc[4], mult[4][4] ;


  w = coord_w1(w1re,w1im) ;  y = coord_y1(y1re,y1im) ;

  z1 = sqrt(2.*w*y*y) ;
  z2 = sqrt(2.*w) ;

  latz1 = lat_z1(z1) ;
  latz2 = lat_z2(z2) ;

  z1dashre = (int) floor(latz1.real()) ;
  z1dashim = (int) floor(latz1.imag()) ;

  z2dashre = (int) floor(latz2.real()) ;
  z2dashim = (int) floor(latz2.imag()) ;

  frc[0] = latz1.real() - ((double) z1dashre) ;
  frc[1] = latz1.imag() - ((double) z1dashim) ;

  frc[2] = latz2.real() - ((double) z2dashre) ;
  frc[3] = latz2.imag() - ((double) z2dashim) ;

  for(i1=0;i1<4;i1++) {
 
    x = frc[i1] ;

    switch(INTERP) {

    case(1) :
      /* 1st Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = 1.-x ;
      mult[i1][2] = x ;
      mult[i1][3] = 0 ;
      break ;

    case(2) :
      /* 2nd Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = (x-1.)*(x-2.)/2. ;
      mult[i1][2] = -x*(x-2.) ;
      mult[i1][3] = x*(x-1)/2. ;
      break ;

    case(3) :
      /* 3rd Order */
      mult[i1][0] = -x*(x-1.)*(x-2.)/6. ;
      mult[i1][1] = ( x*x*x - 2.*x*x - x + 2. )/2. ;
      mult[i1][2] = x*( -x*x + x + 2. )/2. ;
      mult[i1][3] = x*(x*x-1.)/6. ;
      break ;

    default : 
      cout << "Error in INTERP (" << INTERP << ")" << endl ; 
      exit(1) ;

    }

  }


  val = 0. ;

  for(i1=-1;i1<=2;i1++) {
    for(i2=-1;i2<=2;i2++) {
      for(i3=-1;i3<=2;i3++) {
	for(i4=-1;i4<=2;i4++) {

	  val2 = 1. ;
	  
	  switch(i1) {
	    
	  case(-1) : val2 *= mult[0][0] ; break ;
	    
	  case(0) : val2 *= mult[0][1] ; break ;
	    
	  case(1) : val2 *= mult[0][2] ; break ;
	    
	  case(2) : val2 *= mult[0][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i2) {
	    
	  case(-1) : val2 *= mult[1][0] ; break ;
	    
	  case(0) : val2 *= mult[1][1] ; break ;
	    
	  case(1) : val2 *= mult[1][2] ; break ;
	    
	  case(2) : val2 *= mult[1][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i3) {
	    
	  case(-1) : val2 *= mult[2][0] ; break ;
	    
	  case(0) : val2 *= mult[2][1] ; break ;
	    
	  case(1) : val2 *= mult[2][2] ; break ;
	    
	  case(2) : val2 *= mult[2][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i4) {
	    
	  case(-1) : val2 *= mult[3][0] ; break ;
	    
	  case(0) : val2 *= mult[3][1] ; break ;
	    
	  case(1) : val2 *= mult[3][2] ; break ;
	    
	  case(2) : val2 *= mult[3][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }
	  
	  tmpz1re = z1dashre+i1 ;
	  tmpz1im = z1dashim+i2 ;

	  tmpz2re = z2dashre+i3 ;
	  tmpz2im = z2dashim+i4 ;

	  iboundaryZ(&tmpz1re,&tmpz1im) ;
	  iboundaryZ(&tmpz2re,&tmpz2im) ;
	  reflectZ(&tmpz1re,&tmpz1im,&tmpz2re,&tmpz2im) ;

	  if(val2!=0.) {
	    val += val2*K2[K2pos[pos2(tmpz1re,tmpz1im,tmpz2re)]+tmpz2im] ;
	    if( tempK2[K2pos[pos2(tmpz1re,tmpz1im,tmpz2re)]+tmpz2im]==-1) {
	      cout << "Terrible error!" << endl ;
	    }
	  }
 
	}
      }
    }
  }
  
  val -= 0.5*pow(KahlerA,2)*log(abs(z2)*abs(z2)) ;

  return( val ) ;


}



/* Gets point from torus K2 patch, calling interpolation routines if
   necessary.

   This routine also deals with the point falling outside the region;

   Re(z1), Im(z1), Re(z2), Im(z2) < 0.25

   by using the discrete isometry (from the additional lattice action
   together with the other discrete symmeties)

   K(z1,z2) = K(0.25 - z1,z2) + KT = K(0.25i - z1,z2) + KT

   and similarly for z2, where KT is a Kahler transform that can be
   computed just by considering the flat metric on the torus patch
   (with Kahler modulus 'b').

 */

double getK2(int z1rein, int z1imin, int z2rein, int z2imin)
{
  int tmp, z1re, z1im, z2re, z2im ;

  double dK ;

  z1re = z1rein ; z1im = z1imin ; z2re = z2rein ; z2im = z2imin ;

  /* Is point in fundamental domain? */

  /* If Re(z1), Im(z1), Re(z2) or Im(z2) < 0 fix this using discrete
  symmetries of K2 */

  iboundaryZ(&z1re,&z1im) ;
  iboundaryZ(&z2re,&z2im) ;

  /* If Im(z2)<max( Re(z1), Im(z1) ) fix this using discrete symmetry */

  reflectZ(&z1re,&z1im,&z2re,&z2im) ;

  /* If the point falls outside the torus boundaries Re(z1), Im(z1),
     Re(z2), Im(z2) < 0.25, map it back in and add the appropraite
     Kahler transform */

  dK = 0. ;

  if(z1re>=NZ+2 && z1im>=NZ+2) {

    dK += + 0.5*pow(KahlerB,2)*4.*coord_z1(NZ+2-1,0).real()*DZ*(z1re-(NZ+2-1)) ;
    dK += + 0.5*pow(KahlerB,2)*4.*coord_z1(NZ+2-1,0).real()*DZ*(z1im-(NZ+2-1)) ;
    z1re = 2*(NZ+2-1) - z1re ;

    z1im = 2*(NZ+2-1) - z1im ;

  } else if(z1re>=NZ+2) {

    dK += + 0.5*pow(KahlerB,2)*4.*coord_z1(NZ+2-1,0).real()*DZ*(z1re-(NZ+2-1)) ;

    z1re = 2*(NZ+2-1) - z1re ;

    tmp = z1re ; z1re = z1im ; z1im = tmp ;

  } else if(z1im>=NZ+2) {

    dK += + 0.5*pow(KahlerB,2)*4.*coord_z1(NZ+2-1,0).real()*DZ*(z1im-(NZ+2-1)) ;

    z1im = 2*(NZ+2-1) - z1im ;

    tmp = z1re ; z1re = z1im ; z1im = tmp ;

  }

  if(z2re>=NZ+2 && z2im>=NZ+2) {

    dK += + 0.5*pow(KahlerB,2)*4.*coord_z1(NZ+2-1,0).real()*DZ*(z2re-(NZ+2-1)) ;
    dK += + 0.5*pow(KahlerB,2)*4.*coord_z1(NZ+2-1,0).real()*DZ*(z2im-(NZ+2-1)) ;
    z2re = 2*(NZ+2-1) - z2re ;

    z2im = 2*(NZ+2-1) - z2im ;

  } else if(z2re>=NZ+2) {

    dK += + 0.5*pow(KahlerB,2)*4.*coord_z1(NZ+2-1,0).real()*DZ*(z2re-(NZ+2-1)) ;

    z2re = 2*(NZ+2-1) - z2re ;

    tmp = z2re ; z2re = z2im ; z2im = tmp ;

  } else if(z2im>=NZ+2) {

    dK += + 0.5*pow(KahlerB,2)*4.*coord_z1(NZ+2-1,0).real()*DZ*(z2im-(NZ+2-1)) ;

    z2im = 2*(NZ+2-1) - z2im ;

    tmp = z2re ; z2re = z2im ; z2im = tmp ;

  }

  /* Check we are still satisfying Im(z2)<max( Re(z1), Im(z1) ), and
     correct if not */
  reflectZ(&z1re,&z1im,&z2re,&z2im) ;

  /* Now we have mapped back into the fundamental domain either the
     point lies in the torus patch as sigma>=MINSIG or sigma is too
     small and the point really lies in one of the EH patches.

     Depending on sigma, get the point from the K2 array or
     interpolate it from the EH patches K1 and K3 (depending on |y|),
     but remember to add on 'dK' which stores the Kahler transform
     from mapping back within the torus domain above.

  */

  if( abs(coord_z1(z1re,z1im))*abs(coord_z1(z1re,z1im)) + abs(coord_z2(z2re,z2im))*abs(coord_z2(z2re,z2im)) >= MINSIG ) { 

    return( dK + K2[K2pos[pos2(z1re,z1im,z2re)]+z2im] ) ;
    
  } else {

    if(abs(coord_z1(z1re,z1im))>=abs(coord_z2(z2re,z2im))) {

      return( dK + getK2fromK1(z1re,z1im,z2re,z2im) ) ;

    } else {

      return( dK + getK2fromK3(z1re,z1im,z2re,z2im) ) ;

    }

  }

}


/* Same as interpolation routines above: interpolates K2 at a point
   inside the EH patch K1 */


double getK2fromK1(int z1re, int z1im, int z2re, int z2im)
{
  complex<double> z1, z2, w, y, latw, laty ;
  
  int wdashre, wdashim, ydashre, ydashim ;
  int tmpwre, tmpwim, tmpyre, tmpyim ;

  int i1, i2, i3, i4 ;
  double val, val2, oldval ;

  double x, frc[4], mult[4][4] ;


  z1 = coord_z1(z1re,z1im) ;  z2 = coord_z2(z2re,z2im) ;

  w = 0.5*z1*z1 ;

  y = z2/z1 ;

  latw = lat_w1(w,y) ;
  laty = lat_y1(w,y) ;

  wdashre = (int) floor(latw.real()) ;
  wdashim = (int) floor(latw.imag()) ;

  ydashre = (int) floor(laty.real()) ;
  ydashim = (int) floor(laty.imag()) ;

  frc[0] = latw.real() - ((double) wdashre) ;
  frc[1] = latw.imag() - ((double) wdashim) ;

  frc[2] = laty.real() - ((double) ydashre) ;
  frc[3] = laty.imag() - ((double) ydashim) ;

  for(i1=0;i1<4;i1++) {
 
    x = frc[i1] ;

    switch(INTERP) {

    case(1) :
      /* 1st Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = 1.-x ;
      mult[i1][2] = x ;
      mult[i1][3] = 0 ;
      break ;

    case(2) :
      /* 2nd Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = (x-1.)*(x-2.)/2. ;
      mult[i1][2] = -x*(x-2.) ;
      mult[i1][3] = x*(x-1)/2. ;
      break ; 

    case(3) :
      /* 3rd Order */
      mult[i1][0] = -x*(x-1.)*(x-2.)/6. ;
      mult[i1][1] = ( x*x*x - 2.*x*x - x + 2. )/2. ;
      mult[i1][2] = x*( -x*x + x + 2. )/2. ;
      mult[i1][3] = x*(x*x-1.)/6. ;
      break ;

    default : 
      cout << "Error in INTERP (" << INTERP << ")" << endl ; 
      exit(1) ;

    }

  }


  val = 0. ;

  for(i1=-1;i1<=2;i1++) {
    for(i2=-1;i2<=2;i2++) {
      for(i3=-1;i3<=2;i3++) {
	for(i4=-1;i4<=2;i4++) {

	  val2 = 1. ;
	  
	  switch(i1) {
	    
	  case(-1) : val2 *= mult[0][0] ; break ;
	    
	  case(0) : val2 *= mult[0][1] ; break ;
	    
	  case(1) : val2 *= mult[0][2] ; break ;
	    
	  case(2) : val2 *= mult[0][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i2) {
	    
	  case(-1) : val2 *= mult[1][0] ; break ;
	    
	  case(0) : val2 *= mult[1][1] ; break ;
	    
	  case(1) : val2 *= mult[1][2] ; break ;
	    
	  case(2) : val2 *= mult[1][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i3) {
	    
	  case(-1) : val2 *= mult[2][0] ; break ;
	    
	  case(0) : val2 *= mult[2][1] ; break ;
	    
	  case(1) : val2 *= mult[2][2] ; break ;
	    
	  case(2) : val2 *= mult[2][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i4) {
	    
	  case(-1) : val2 *= mult[3][0] ; break ;
	    
	  case(0) : val2 *= mult[3][1] ; break ;
	    
	  case(1) : val2 *= mult[3][2] ; break ;
	    
	  case(2) : val2 *= mult[3][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  tmpwre = wdashre+i1 ;
	  tmpwim = wdashim+i2 ;

	  tmpyre = ydashre+i3 ;
	  tmpyim = ydashim+i4 ;

	  iboundaryWY(&tmpwre,&tmpwim,&tmpyre,&tmpyim) ;

	  if(val2!=0.) {
	    val += val2*K1[pos1(tmpwre,tmpwim,tmpyre,tmpyim)] ;
	    if(tempK1[pos1(tmpwre,tmpwim,tmpyre,tmpyim)]==-1) {
	      cout << "Terrible error!" << endl ;
	    }
	  }

	}
      }
    }
  }
  
  val += 0.5*pow(KahlerA,2)*log(2.*abs(w)) ;

  return( val ) ;


}


/* As above, but interpolates K2 from K3 EH patch */

double getK2fromK3(int z1re, int z1im, int z2re, int z2im)
{
  complex<double> z1, z2, w, y, latw, laty ;
  
  int wdashre, wdashim, ydashre, ydashim ;
  int tmpwre, tmpwim, tmpyre, tmpyim ;
  
  int i1, i2, i3, i4 ;
  double val, val2, oldval ;

  double x, frc[4], mult[4][4] ;


  z1 = coord_z1(z1re,z1im) ;  z2 = coord_z2(z2re,z2im) ;

  w = 0.5*z2*z2 ;

  y = z1/z2 ;

  latw = lat_w1(w,y) ;
  laty = lat_y1(w,y) ;

  wdashre = (int) floor(latw.real()) ;
  wdashim = (int) floor(latw.imag()) ;

  ydashre = (int) floor(laty.real()) ;
  ydashim = (int) floor(laty.imag()) ;

  frc[0] = latw.real() - ((double) wdashre) ;
  frc[1] = latw.imag() - ((double) wdashim) ;

  frc[2] = laty.real() - ((double) ydashre) ;
  frc[3] = laty.imag() - ((double) ydashim) ;

  for(i1=0;i1<4;i1++) {
 
    x = frc[i1] ;

    switch(INTERP) {

    case(1) :
      /* 1st Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = 1.-x ;
      mult[i1][2] = x ;
      mult[i1][3] = 0 ;
      break ;

    case(2) :
      /* 2nd Order */
      mult[i1][0] = 0 ;
      mult[i1][1] = (x-1.)*(x-2.)/2. ;
      mult[i1][2] = -x*(x-2.) ;
      mult[i1][3] = x*(x-1)/2. ;
      break ;

    case(3) :
      /* 3rd Order */
      mult[i1][0] = -x*(x-1.)*(x-2.)/6. ;
      mult[i1][1] = ( x*x*x - 2.*x*x - x + 2. )/2. ;
      mult[i1][2] = x*( -x*x + x + 2. )/2. ;
      mult[i1][3] = x*(x*x-1.)/6. ;
      break ;

    default : 
      cout << "Error in INTERP (" << INTERP << ")" << endl ; 
      exit(1) ;

    }

  }

  val = 0. ;

  for(i1=-1;i1<=2;i1++) {
    for(i2=-1;i2<=2;i2++) {
      for(i3=-1;i3<=2;i3++) {
	for(i4=-1;i4<=2;i4++) {

	  val2 = 1. ;
	  
	  switch(i1) {
	    
	  case(-1) : val2 *= mult[0][0] ; break ;
	    
	  case(0) : val2 *= mult[0][1] ; break ;
	    
	  case(1) : val2 *= mult[0][2] ; break ;
	    
	  case(2) : val2 *= mult[0][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i2) {
	    
	  case(-1) : val2 *= mult[1][0] ; break ;
	    
	  case(0) : val2 *= mult[1][1] ; break ;
	    
	  case(1) : val2 *= mult[1][2] ; break ;
	    
	  case(2) : val2 *= mult[1][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i3) {
	    
	  case(-1) : val2 *= mult[2][0] ; break ;
	    
	  case(0) : val2 *= mult[2][1] ; break ;
	    
	  case(1) : val2 *= mult[2][2] ; break ;
	    
	  case(2) : val2 *= mult[2][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  switch(i4) {
	    
	  case(-1) : val2 *= mult[3][0] ; break ;
	    
	  case(0) : val2 *= mult[3][1] ; break ;
	    
	  case(1) : val2 *= mult[3][2] ; break ;
	    
	  case(2) : val2 *= mult[3][3] ; break ;

	  default : cout << "Terrible error!" << endl ; exit(1) ;

	  }

	  tmpwre = wdashre+i1 ;
	  tmpwim = wdashim+i2 ;

	  tmpyre = ydashre+i3 ;
	  tmpyim = ydashim+i4 ;

	  iboundaryWY(&tmpwre,&tmpwim,&tmpyre,&tmpyim) ;

	  if(val2!=0.) {
	    val += val2*K1[pos1(tmpwre,tmpwim,tmpyre,tmpyim)] ;
	    if(tempK1[pos1(tmpwre,tmpwim,tmpyre,tmpyim)]==-1) {
	      cout << "Terrible error!" << endl ;
	    }
	  }

	}
      }
    }
  }
  
  val += 0.5*pow(KahlerA,2)*log(2.*abs(w)) ;

  return( val ) ;


}

